<?php
/**
 * admin_bootstrap.php
 *
 * Single bootstrap file for all admin pages.
 * Include this file at the top of any admin page to automatically load:
 * - Constants (STORAGE_DIR, etc.)
 * - Session configuration
 * - Authentication check
 * - Policy system
 * - Helper functions
 * - Database connection
 * - CSRF token initialization
 *
 * Usage:
 *   require_once __DIR__ . '/../includes/admin_bootstrap.php';
 */

// Constants (must be first - defines STORAGE_DIR, etc.)
require_once __DIR__ . '/constants.php';

// Session configuration
require_once __DIR__ . '/session_config.php';

// Start session if not already started
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

// Authentication check
require_once __DIR__ . '/auth_check.php';

// Policy system (role-based permissions)
require_once __DIR__ . '/policy.php';

// Helper functions (URL generation, CSRF, etc.)
require_once __DIR__ . '/helpers.php';

// Settings system (app-wide configuration)
require_once __DIR__ . '/settings.php';

// Database connection
require_once __DIR__ . '/../config/db.php';

// Ensure user is logged in
require_login();

// Initialize CSRF token if not exists
if (empty($_SESSION['csrf'])) {
    $_SESSION['csrf'] = bin2hex(random_bytes(16));
}

// Make CSRF token available globally
$GLOBALS['csrf'] = $_SESSION['csrf'];

/**
 * Get the current CSRF token
 * 
 * @return string The CSRF token
 */
function get_csrf_token(): string {
    return $GLOBALS['csrf'] ?? '';
}

/**
 * Validate CSRF token from POST request
 * 
 * @param string|null $token The token to validate (defaults to $_POST['csrf'])
 * @return bool True if valid, false otherwise
 */
function validate_csrf_token(?string $token = null): bool {
    $token = $token ?? ($_POST['csrf'] ?? '');
    return hash_equals(get_csrf_token(), $token);
}

/**
 * Check if current request is a POST request with valid CSRF
 * 
 * @return bool True if POST with valid CSRF
 */
function is_valid_post(): bool {
    return $_SERVER['REQUEST_METHOD'] === 'POST' && validate_csrf_token();
}

/**
 * Get current user role (convenience function)
 * 
 * @return string The current user's role
 */
function get_current_role(): string {
    return current_user_role();
}

/**
 * Check if current user has specific permission
 * 
 * @param callable $permission_check The permission check function
 * @return bool True if has permission
 */
function can(callable $permission_check): bool {
    return $permission_check(get_current_role());
}
