<?php
/**
 * driver_bootstrap.php
 *
 * Single bootstrap file for all driver pages.
 * Include this file at the top of any driver page to automatically load:
 * - Constants (STORAGE_DIR, etc.)
 * - Session configuration
 * - Driver authentication check
 * - Helper functions
 * - Database connection
 * - CSRF token initialization
 *
 * Usage:
 *   require_once __DIR__ . '/../includes/driver_bootstrap.php';
 */

// Constants (must be first - defines STORAGE_DIR, etc.)
require_once __DIR__ . '/constants.php';

// Session configuration
require_once __DIR__ . '/session_config.php';

// Start session if not already started
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

// Driver authentication
require_once __DIR__ . '/driver_auth.php';

// Helper functions (URL generation, CSRF, etc.)
require_once __DIR__ . '/helpers.php';

// Settings system (app-wide configuration)
require_once __DIR__ . '/settings.php';

// Database connection
require_once __DIR__ . '/../config/db.php';

// Time helpers (for today_local, yesterday_local functions)
require_once __DIR__ . '/time.php';

// Ensure driver is logged in
require_driver_login();

// Initialize CSRF token if not exists
if (empty($_SESSION['csrf'])) {
    $_SESSION['csrf'] = bin2hex(random_bytes(16));
}

// Make CSRF token available globally
$GLOBALS['csrf'] = $_SESSION['csrf'];

/**
 * Get the current CSRF token
 * 
 * @return string The CSRF token
 */
function get_csrf_token(): string {
    return $GLOBALS['csrf'] ?? '';
}

/**
 * Validate CSRF token from POST request
 * 
 * @param string|null $token The token to validate (defaults to $_POST['csrf'])
 * @return bool True if valid, false otherwise
 */
function validate_csrf_token(?string $token = null): bool {
    $token = $token ?? ($_POST['csrf'] ?? '');
    return hash_equals(get_csrf_token(), $token);
}

/**
 * Check if current request is a POST request with valid CSRF
 * 
 * @return bool True if POST with valid CSRF
 */
function is_valid_post(): bool {
    return $_SERVER['REQUEST_METHOD'] === 'POST' && validate_csrf_token();
}

/**
 * Get current driver ID (convenience function)
 * 
 * @return int The current driver's ID
 */
function get_current_driver_id(): int {
    return current_driver_id();
}

/**
 * Get current driver's partner ID (convenience function)
 * 
 * @return string|null The current driver's partner ID
 */
function get_current_partner_id(): ?string {
    return current_driver_partner_id();
}
