<?php
// includes/id_generator.php
// Generates next partner_id like AK0001 per project using atomic counter.

function generate_partner_id(mysqli $conn, int $project_id): ?string {
  // Start transaction
  $conn->begin_transaction();

  try {
    // Lock the project row for consistent prefix read
    $stmt = $conn->prepare('SELECT prefix FROM projects WHERE id=? FOR UPDATE');
    $stmt->bind_param('i', $project_id);
    $stmt->execute();
    $res = $stmt->get_result();
    if (!$row = $res->fetch_assoc()) {
      $conn->rollback();
      return null;
    }
    $prefix = strtoupper(trim($row['prefix'] ?? ''));
    if ($prefix === '') {
      $conn->rollback();
      return null;
    }

    // Insert or increment counter atomically
    // The LAST_INSERT_ID trick returns the new value reliably
    $sql = 'INSERT INTO project_counters(project_id,last_no) VALUES(?,1)
            ON DUPLICATE KEY UPDATE last_no=LAST_INSERT_ID(last_no+1)';
    $stmt2 = $conn->prepare($sql);
    $stmt2->bind_param('i', $project_id);
    $stmt2->execute();

    $next = (int)$conn->insert_id; // LAST_INSERT_ID value

    // Format: PREFIX + 4-digit zero padded number
    $partner_id = $prefix . str_pad((string)$next, 4, '0', STR_PAD_LEFT);

    $conn->commit();
    return $partner_id;
  } catch (Throwable $e) {
    $conn->rollback();
    return null;
  }
}