<?php
require_once __DIR__ . '/../../includes/admin_bootstrap.php';

// Use a policy function to check for admin/superadmin roles.
if (!can_manage_assets(get_current_role())) {
    http_response_code(403);
    exit('Forbidden');
}

// --- SETUP ---
if (session_status() === PHP_SESSION_NONE) session_start();
if (empty($_SESSION['csrf'])) $_SESSION['csrf'] = bin2hex(random_bytes(16));
$csrf = $_SESSION['csrf'];

$title = 'Asset Catalog Management';
$err = '';
$form_data = $_SESSION['form_data'] ?? [];
unset($_SESSION['form_data']);

// --- POST HANDLER ---
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['csrf']) && hash_equals($csrf, $_POST['csrf'])) {
    $asset_id = isset($_POST['asset_id']) && ctype_digit($_POST['asset_id']) ? (int)$_POST['asset_id'] : 0;
    $name = trim($_POST['asset_name']);
    $category = trim($_POST['asset_category']);
    $has_expiry = isset($_POST['has_expiry']) ? 1 : 0;
    $lifespan_days = !empty($_POST['lifespan_days']) ? (int)$_POST['lifespan_days'] : null;
    $is_active = isset($_POST['is_active']) ? 1 : 0;

    if (!$has_expiry) {
        $lifespan_days = null;
    }

    if (empty($name) || empty($category)) {
        $err = 'Asset Name and Category are required.';
        $_SESSION['form_data'] = $_POST; // Persist data on validation error
    } else {
        if ($asset_id > 0) { // Update existing asset
            $stmt = $conn->prepare("UPDATE asset_items SET name=?, category=?, has_expiry=?, lifespan_days=?, is_active=? WHERE id=?");
            $stmt->bind_param('ssiiii', $name, $category, $has_expiry, $lifespan_days, $is_active, $asset_id);
            if ($stmt->execute()) {
                $_SESSION['msg'] = ['text' => 'Asset updated successfully.', 'type' => 'success'];
            } else {
                $_SESSION['msg'] = ['text' => 'Failed to update asset.', 'type' => 'danger'];
            }
        } else { // Insert new asset
            $stmt = $conn->prepare("INSERT INTO asset_items (name, category, has_expiry, lifespan_days, is_active) VALUES (?, ?, ?, ?, ?)");
            $stmt->bind_param('ssiii', $name, $category, $has_expiry, $lifespan_days, $is_active);
            if ($stmt->execute()) {
                $_SESSION['msg'] = ['text' => 'Asset created successfully.', 'type' => 'success'];
            } else {
                $_SESSION['msg'] = ['text' => 'Failed to create asset.', 'type' => 'danger'];
            }
        }
        // Redirect to prevent duplicate submission on refresh
        header("Location: " . $_SERVER['REQUEST_URI']);
        exit();
    }
}

// --- VIEW DATA ---
$msg = '';
$msg_type = '';
if (isset($_SESSION['msg'])) {
    $msg = $_SESSION['msg']['text'];
    $msg_type = $_SESSION['msg']['type'];
    unset($_SESSION['msg']);
}

// Fetch all assets for display
$assets = $conn->query("
    SELECT a.*, p.name AS project_name 
    FROM asset_items a 
    LEFT JOIN projects p ON a.project_id = p.id 
    ORDER BY a.category, a.name
")->fetch_all(MYSQLI_ASSOC);
$projects = $conn->query("SELECT id, name FROM projects ORDER BY name")->fetch_all(MYSQLI_ASSOC);
?>
<!doctype html>
<html lang="en">
<head>
  <meta charset="utf-8">
  <title><?php echo $title; ?> | Hisbuu</title>
  <meta name="viewport" content="width=device-width, initial-scale=1">
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
  <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.3/font/bootstrap-icons.css" rel="stylesheet">
</head>
<body class="bg-light">
<?php include __DIR__ . '/../../includes/admin_nav.php'; ?>

<div class="container py-4">
    <div class="d-flex justify-content-between align-items-center mb-3">
        <h5 class="mb-0"><?php echo $title; ?></h5>
        <button class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#assetModal" data-action="add">
            <i class="bi bi-plus-circle"></i> Add New Asset
        </button>
    </div>

    <?php if($msg): ?>
        <div class="alert alert-<?php echo $msg_type; ?> alert-dismissible fade show" role="alert">
            <?php echo $msg; ?>
            <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
        </div>
    <?php endif; ?>
    <?php if($err): ?>
        <div class="alert alert-danger"><?php echo $err; ?></div>
    <?php endif; ?>


    <div class="card">
        <div class="card-body">
            <div class="table-responsive">
                <table class="table table-striped">
                    <thead>
                        <tr>
                            <th>Name</th>
                            <th>Category</th>
                        <th>Project</th>
                            <th>Has Expiry?</th>
                            <th>Lifespan (Days)</th>
                            <th>Status</th>
                            <th>Action</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php if (!$assets): ?>
                        <tr><td colspan="7" class="text-center text-muted">No asset items found. Add one to get started.</td></tr>
                        <?php else: foreach ($assets as $asset): ?>
                            <tr>
                                <td><?php echo e($asset['name']); ?></td>
                                <td><?php echo e($asset['category']); ?></td>
                            <td><?php echo $asset['project_name'] ? e($asset['project_name']) : '<span class="text-muted">Global</span>'; ?></td>
                                <td><?php echo $asset['has_expiry'] ? 'Yes' : 'No'; ?></td>
                                <td><?php echo $asset['lifespan_days'] ?? 'N/A'; ?></td>
                                <td><?php echo $asset['is_active'] ? '<span class="badge text-bg-success">Active</span>' : '<span class="badge text-bg-secondary">Inactive</span>'; ?></td>
                                <td>
                                    <button class="btn btn-sm btn-outline-secondary edit-btn"
                                            data-bs-toggle="modal"
                                            data-bs-target="#assetModal"
                                            data-action="edit"
                                            data-asset='<?php echo htmlspecialchars(json_encode($asset), ENT_QUOTES, 'UTF-8'); ?>'>
                                        Edit
                                    </button>
                                </td>
                            </tr>
                        <?php endforeach; endif; ?>
                    </tbody>
                </table>
            </div>
        </div>
    </div>

    <!-- Add/Edit Modal -->
    <div class="modal fade" id="assetModal" tabindex="-1" aria-labelledby="assetModalLabel" aria-hidden="true">
        <div class="modal-dialog">
            <div class="modal-content">
                <form method="post">
                    <input type="hidden" name="csrf" value="<?php echo $csrf; ?>">
                    <div class="modal-header">
                        <h5 class="modal-title" id="assetModalLabel">Add Asset Item</h5>
                        <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                    </div>
                    <div class="modal-body">
                        <input type="hidden" name="asset_id" id="asset_id" value="<?php echo e($form_data['asset_id'] ?? ''); ?>">
                        <div class="mb-3">
                            <label for="asset_name" class="form-label">Asset Name</label>
                            <input type="text" class="form-control" id="asset_name" name="asset_name" placeholder="e.g., Rider T-Shirt - L" value="<?php echo e($form_data['asset_name'] ?? ''); ?>" required>
                        </div>
                        <div class="mb-3">
                            <label for="asset_category" class="form-label">Category</label>
                            <input type="text" class="form-control" id="asset_category" name="asset_category" placeholder="e.g., Apparel, Safety Gear" value="<?php echo e($form_data['asset_category'] ?? ''); ?>" required>
                        </div>
                    <div class="mb-3">
                        <label for="project_id" class="form-label">Assign to Project</label>
                        <select class="form-select" id="project_id" name="project_id">
                            <option value="">Global (All Projects)</option>
                            <?php foreach ($projects as $project): ?>
                                <option value="<?php echo $project['id']; ?>"><?php echo e($project['name']); ?></option>
                            <?php endforeach; ?>
                        </select>
                        <div class="form-text">Assigning to a project will make this asset primarily visible for that project's drivers.</div>
                    </div>
                        <div class="form-check mb-3">
                            <input class="form-check-input" type="checkbox" value="1" id="has_expiry" name="has_expiry" <?php echo isset($form_data['has_expiry']) ? 'checked' : ''; ?>>
                            <label class="form-check-label" for="has_expiry">This item has an expiry date</label>
                        </div>
                        <div class="mb-3" id="lifespan_div" style="display: none;">
                            <label for="lifespan_days" class="form-label">Default Lifespan (in days)</label>
                            <input type="number" class="form-control" id="lifespan_days" name="lifespan_days" placeholder="e.g., 365" value="<?php echo e($form_data['lifespan_days'] ?? ''); ?>">
                        </div>
                        <div class="form-check form-switch">
                            <input class="form-check-input" type="checkbox" role="switch" id="is_active" name="is_active" value="1" <?php echo isset($form_data['is_active']) || empty($form_data) ? 'checked' : ''; ?>>
                            <label class="form-check-label" for="is_active">Is Active (can be assigned)</label>
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                        <button type="submit" class="btn btn-primary">Save changes</button>
                    </div>
                </form>
            </div>
        </div>
    </div>
</div>

<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"></script>
<script>
document.addEventListener('DOMContentLoaded', function () {
    const assetModal = document.getElementById('assetModal');
    const hasExpiryCheck = document.getElementById('has_expiry');
    const lifespanDiv = document.getElementById('lifespan_div');

    function toggleLifespan() {
        lifespanDiv.style.display = hasExpiryCheck.checked ? 'block' : 'none';
    }
    hasExpiryCheck.addEventListener('change', toggleLifespan);
    
    <?php
    if ($err && !empty($form_data)) {
        echo "const modal = new bootstrap.Modal(assetModal); modal.show(); toggleLifespan();";
    }
    ?>

    assetModal.addEventListener('show.bs.modal', function (event) {
        const button = event.relatedTarget;
        if (!button) return;
        
        const action = button.getAttribute('data-action');
        const modalTitle = assetModal.querySelector('.modal-title');
        const form = assetModal.querySelector('form');

        form.reset();
        document.getElementById('asset_id').value = '';
        toggleLifespan();

        if (action === 'edit') {
            modalTitle.textContent = 'Edit Asset Item';
            const asset = JSON.parse(button.getAttribute('data-asset'));
            
            document.getElementById('asset_id').value = asset.id;
            document.getElementById('asset_name').value = asset.name;
            document.getElementById('asset_category').value = asset.category;
            document.getElementById('project_id').value = asset.project_id || '';
            document.getElementById('is_active').checked = !!parseInt(asset.is_active);
            document.getElementById('has_expiry').checked = !!parseInt(asset.has_expiry);
            
            if (!!parseInt(asset.has_expiry)) {
                document.getElementById('lifespan_days').value = asset.lifespan_days;
            }
        } else {
            modalTitle.textContent = 'Add Asset Item';
            document.getElementById('is_active').checked = true;
        }
        toggleLifespan();
    });
});
</script>
</body>
</html>
