<?php
require_once __DIR__ . '/../../includes/admin_bootstrap.php';

// Use the same policy function to protect this page
if (!can_manage_assets(current_user_role())) {
    http_response_code(403);
    exit('Forbidden');
}

if (empty($_SESSION['csrf'])) $_SESSION['csrf'] = bin2hex(random_bytes(16));
$csrf = $_SESSION['csrf'];

$title = 'Assign Assets to Drivers';

// Handle POST for assignment
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['csrf']) && hash_equals($csrf, $_POST['csrf'])) {
    $driver_ids = $_POST['driver_ids'] ?? [];
    $asset_quantities = $_POST['asset_quantities'] ?? [];
    $selected_assets = array_filter($asset_quantities, fn($qty) => (int)$qty > 0);
    
    if (empty($driver_ids) || empty($selected_assets)) {
        $_SESSION['msg'] = ['text' => 'You must select at least one driver and one asset.', 'type' => 'warning'];
    } else {
        // Get the admin/supervisor's ID directly from the session to avoid function-not-found errors.
        $assigned_by = (int)($_SESSION['user']['id'] ?? 0);

        $asset_ids_to_fetch = array_keys($selected_assets);
        $assets_to_assign = $conn->query("SELECT id, has_expiry, lifespan_days FROM asset_items WHERE is_active=1 AND id IN (".implode(',', array_map('intval', $asset_ids_to_fetch)).")")->fetch_all(MYSQLI_ASSOC);

        if ($assets_to_assign) {
            $stmt = $conn->prepare("INSERT INTO driver_assets (driver_id, asset_item_id, assigned_by_user_id, expiry_date) VALUES (?, ?, ?, ?)");

            $count = 0;
            foreach ($driver_ids as $driver_id) {
                foreach ($assets_to_assign as $asset) {
                    $quantity = (int)($selected_assets[$asset['id']] ?? 0);
                    for ($i = 0; $i < $quantity; $i++) {
                        $expiry_date = null;
                        if ($asset['has_expiry'] && $asset['lifespan_days']) {
                            $expiry_date = date('Y-m-d', strtotime("+{$asset['lifespan_days']} days"));
                        }
                        $stmt->bind_param('iiis', $driver_id, $asset['id'], $assigned_by, $expiry_date);
                        $stmt->execute();
                        $count++;
                    }
                }
            }
            $stmt->close();
            
            $_SESSION['msg'] = ['text' => "Successfully assigned assets to " . count($driver_ids) . " driver(s). Total items assigned: " . $count, 'type' => 'success'];
        } else {
            $_SESSION['msg'] = ['text' => 'No valid, active assets were selected for assignment.', 'type' => 'warning'];
        }
    }
    
    header("Location: " . $_SERVER['REQUEST_URI']);
    exit();
}

// Check for session message
$msg = '';
$msg_type = '';
if (isset($_SESSION['msg'])) {
    $msg = $_SESSION['msg']['text'];
    $msg_type = $_SESSION['msg']['type'];
    unset($_SESSION['msg']);
}

// Fetch active drivers and active assets for the form
$projects = $conn->query("SELECT id, name FROM projects ORDER BY name")->fetch_all(MYSQLI_ASSOC);

$selected_project_id = isset($_GET['project_id']) && ctype_digit($_GET['project_id']) ? (int)$_GET['project_id'] : null;

// Filter drivers by project if selected
$driver_sql = "SELECT id, partner_id, first_name, last_name FROM drivers ORDER BY first_name, last_name";
if ($selected_project_id) {
    $driver_sql = "SELECT id, partner_id, first_name, last_name FROM drivers WHERE project_id = $selected_project_id ORDER BY first_name, last_name";
}
$drivers = $conn->query($driver_sql)->fetch_all(MYSQLI_ASSOC);

// Filter assets by project if selected (show project-specific AND global assets)
$asset_sql = "SELECT id, name, category FROM asset_items WHERE is_active=1 ORDER BY category, name";
if ($selected_project_id) {
    $asset_sql = "SELECT id, name, category FROM asset_items WHERE is_active=1 AND (project_id = $selected_project_id OR project_id IS NULL) ORDER BY category, name";
}
$assets = $conn->query($asset_sql)->fetch_all(MYSQLI_ASSOC);
?>
<!doctype html>
<html lang="en">
<head>
  <meta charset="utf-8">
  <title><?php echo $title; ?> | Hisbuu</title>
  <meta name="viewport" content="width=device-width, initial-scale=1">
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
  <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.3/font/bootstrap-icons.css" rel="stylesheet">
</head>
<body class="bg-light">
<?php include __DIR__ . '/../../includes/admin_nav.php'; ?>

<div class="container py-4">
    <div class="d-flex justify-content-between align-items-center mb-3">
        <h5 class="mb-0"><?php echo $title; ?></h5>
    </div>
    
    <div class="card mb-3">
        <div class="card-body">
            <form method="get" class="row align-items-end">
                <div class="col-md-4">
                    <label for="project_id" class="form-label">Filter by Project</label>
                    <select class="form-select" id="project_id" name="project_id" onchange="this.form.submit()">
                        <option value="">All Projects</option>
                        <?php foreach ($projects as $project): ?>
                            <option value="<?php echo $project['id']; ?>" <?php echo $selected_project_id == $project['id'] ? 'selected' : ''; ?>><?php echo e($project['name']); ?></option>
                        <?php endforeach; ?>
                    </select>
                </div>
            </form>
        </div>
    </div>

    <?php if($msg): ?>
        <div class="alert alert-<?php echo $msg_type; ?> alert-dismissible fade show" role="alert">
            <?php echo $msg; ?>
            <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
        </div>
    <?php endif; ?>

    <form method="post" onsubmit="return confirm('Are you sure you want to assign the selected assets to the selected drivers?');" id="assign-form">
        <input type="hidden" name="csrf" value="<?php echo $csrf; ?>">
        <div class="row">
            <!-- Driver Selection -->
            <div class="col-lg-5">
                <div class="card">
                    <div class="card-header d-flex justify-content-between align-items-center">
                        <h6 class="mb-0">1. Select Drivers</h6>
                        <button type="button" class="btn btn-sm btn-outline-secondary" onclick="toggleAll(this, 'driver_ids[]')">Select All</button>
                    </div>
                    <div class="card-body" style="max-height: 500px; overflow-y: auto;">
                        <?php if (!$drivers): ?>
                            <p class="text-muted">No active drivers found.</p>
                        <?php else: ?>
                            <ul class="list-group list-group-flush">
                            <?php foreach ($drivers as $driver): ?>
                                <li class="list-group-item">
                                    <div class="form-check">
                                        <input class="form-check-input" type="checkbox" name="driver_ids[]" value="<?php echo $driver['id']; ?>" id="driver_<?php echo $driver['id']; ?>">
                                        <label class="form-check-label" for="driver_<?php echo $driver['id']; ?>">
                                            <?php echo e($driver['first_name'] . ' ' . $driver['last_name']); ?>
                                            <small class="text-muted">(ID: <?php echo e($driver['partner_id']); ?>)</small>
                                        </label>
                                    </div>
                                </li>
                            <?php endforeach; ?>
                            </ul>
                        <?php endif; ?>
                    </div>
                </div>
            </div>

            <!-- Asset Selection -->
            <div class="col-lg-7 mt-3 mt-lg-0">
                <div class="card">
                    <div class="card-header d-flex justify-content-between align-items-center">
                        <h6 class="mb-0">2. Select Assets & Quantities</h6>
                    </div>
                    <div class="card-body" style="max-height: 500px; overflow-y: auto;">
                        <?php if (!$assets): ?>
                            <p class="text-muted">No active assets found. Please add items to the <a href="assets.php">Asset Catalog</a> first.</p>
                        <?php else: ?>
                            <table class="table table-sm">
                            <?php foreach ($assets as $asset): ?>
                                <tr>
                                    <td>
                                        <label for="asset_<?php echo $asset['id']; ?>">
                                            <?php echo e($asset['name']); ?>
                                            <small class="text-muted">(<?php echo e($asset['category']); ?>)</small>
                                        </label>
                                    </td>
                                    <td style="width: 100px;">
                                        <input type="number" class="form-control form-control-sm" name="asset_quantities[<?php echo $asset['id']; ?>]" id="asset_<?php echo $asset['id']; ?>" min="0" value="0" placeholder="Qty">
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                            </table>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        </div>

        <div class="d-grid mt-3">
            <button type="submit" class="btn btn-primary btn-lg" id="submit-btn">Assign Selected Assets to Drivers</button>
        </div>
    </form>
</div>

<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"></script>
<script>
function toggleAll(button, name) {
    const checkboxes = document.getElementsByName(name);
    const isSelectAll = button.textContent === 'Select All';
    for (let i = 0; i < checkboxes.length; i++) {
        checkboxes[i].checked = isSelectAll;
    }
    button.textContent = isSelectAll ? 'Deselect All' : 'Select All';
}

document.getElementById('assign-form').addEventListener('submit', function(e) {
    const driverChecks = document.querySelectorAll('input[name="driver_ids[]"]:checked').length;
    const assetChecks = Array.from(document.querySelectorAll('input[name^="asset_quantities"]')).some(input => parseInt(input.value) > 0);
    if (driverChecks === 0 || assetChecks === 0) {
        alert('Please select at least one driver and one asset.');
        e.preventDefault();
        return false;
    }
});
</script>
</body>
</html>
