<?php
require_once __DIR__ . '/../../includes/admin_bootstrap.php';

function download_template(): void {
  header('Content-Type: text/csv; charset=utf-8');
  header('Content-Disposition: attachment; filename="bank_transactions_template.csv"');
  $out = fopen('php://output', 'w');
  fputcsv($out, ['trx_no','amount','date']);
  fputcsv($out, ['LFT200300400','22.356','2025-10-24']);
  fputcsv($out, ['LFT200300401','18.000','2025-10-24']);
  fclose($out);
  exit;
}
// Handle template download before any auth checks
if (isset($_GET['action']) && $_GET['action'] === 'template') download_template();

require_login();
$role = current_user_role();
if (!can_manage_finance($role) && !in_array($role, ['Administrator','Super Administrator'], true)) {
  http_response_code(403);
  exit('Forbidden');
}

if (session_status() === PHP_SESSION_NONE) session_start();
if (empty($_SESSION['csrf'])) $_SESSION['csrf'] = bin2hex(random_bytes(16));
$csrf = $_SESSION['csrf'];

header('X-Frame-Options: SAMEORIGIN');

function hmac_sign(string $data, string $key): string {
  return hash_hmac('sha256', $data, $key);
}

$flash_ok = '';
$flash_err = '';
$is_preview = ($_SERVER['REQUEST_METHOD'] === 'POST' && ($_POST['phase'] ?? '') === 'preview');
$is_import  = ($_SERVER['REQUEST_METHOD'] === 'POST' && ($_POST['phase'] ?? '') === 'import');

$preview_rows = [];
$ok_rows = [];
$stats = ['total'=>0,'ok'=>0,'bad'=>0];
$has_header = isset($_POST['has_header']) ? (bool)$_POST['has_header'] : true;

// Phase 2: confirmed import
if ($is_import && isset($_POST['csrf']) && hash_equals($csrf, $_POST['csrf'])) {
  $payload = trim($_POST['payload'] ?? '');
  $sig     = trim($_POST['sig'] ?? '');
  if ($payload === '' || $sig === '') {
    $flash_err = 'Missing import payload.';
  } else {
    $expected = hmac_sign($payload, $csrf);
    if (!hash_equals($expected, $sig)) {
      $flash_err = 'Invalid or expired token.';
    } else {
      $rows = json_decode($payload, true);
      if (!is_array($rows)) {
        $flash_err = 'Payload malformed.';
      } else {
        $stmt = $conn->prepare("
          UPDATE submissions
          SET bank_trx_no = CASE
                WHEN bank_trx_no IS NULL OR bank_trx_no=''
                  THEN ?
                ELSE bank_trx_no
              END,
              bank_validated = 1,
              is_trx_validated = 1,
              bank_validated_at = NOW(),
              bank_amount = ?,
              bank_date = ?,
              bank_note = 'Matched via bank upload'
          WHERE id = ?
        ");
        if (!$stmt) {
          $flash_err = 'Cannot prepare import statement.';
        } else {
          $matched = 0; $failed = 0;
          foreach ($rows as $row) {
            $trx    = (string)($row['trx_no'] ?? '');
            $amount = (float)($row['amount'] ?? 0);
            $date   = trim((string)($row['date'] ?? ''));
            $id     = (int)($row['submission_id'] ?? 0);
            if ($trx === '' || $id <= 0) { $failed++; continue; }
            $stmt->bind_param('sdsi', $trx, $amount, $date, $id);
            if ($stmt->execute() && $stmt->affected_rows > 0) $matched++; else $failed++;
          }
          $flash_ok = "Updated $matched rows. Skipped $failed.";
        }
      }
    }
  }
}

// Phase 1: preview
if ($is_preview && isset($_POST['csrf']) && hash_equals($csrf, $_POST['csrf'])) {
  if (!isset($_FILES['bank_csv']) || $_FILES['bank_csv']['error'] !== UPLOAD_ERR_OK) {
    $flash_err = 'Upload failed. Provide a valid CSV.';
  } else {
    $fh = fopen($_FILES['bank_csv']['tmp_name'], 'r');
    if (!$fh) {
      $flash_err = 'Cannot open uploaded file.';
    } else {
      $rownum = 0;
      $cache = [];
      $lookup = $conn->prepare("
        SELECT s.id, s.service_date, s.trx_no, s.bank_trx_no,
               d.partner_id, CONCAT(COALESCE(d.first_name,''),' ',COALESCE(d.last_name,'')) AS driver_name
        FROM submissions s
        JOIN drivers d ON d.id = s.driver_id
        WHERE s.trx_no=? OR s.bank_trx_no=?
        LIMIT 1
      ");
      if (!$lookup) {
        $flash_err = 'Database error while preparing lookup.';
      } else while (($cols = fgetcsv($fh)) !== false) {
        $rownum++;
        if ($has_header && $rownum === 1) continue;

        $stats['total']++;
        $cols = array_pad($cols, 3, '');
        list($trx_raw, $amount_raw, $date_raw) = array_map('trim', $cols);

        $errors = [];
        if ($trx_raw === '') $errors[] = 'trx_no required';

        $amount_ok = $amount_raw !== '' && preg_match('/^-?\d+(?:\.\d{1,3})?$/', $amount_raw);
        if (!$amount_ok) $errors[] = 'amount invalid';
        $amount_val = $amount_ok ? (float)$amount_raw : 0.0;

        $date_val = '';
        if ($date_raw !== '') {
          $dt = DateTime::createFromFormat('Y-m-d', $date_raw);
          if (!$dt || $dt->format('Y-m-d') !== $date_raw) {
            $errors[] = 'date must be YYYY-MM-DD';
          } else {
            $date_val = $date_raw;
          }
        }

        $match = null;
        if (!$errors && $trx_raw !== '') {
          if (!isset($cache[$trx_raw])) {
            $lookup->bind_param('ss', $trx_raw, $trx_raw);
            $lookup->execute();
            $cache[$trx_raw] = $lookup->get_result()->fetch_assoc() ?: null;
          }
          $match = $cache[$trx_raw];
          if (!$match) {
            $errors[] = 'no submission found with this trx';
          }
        }

        $status = $errors ? 'error' : 'ok';
        if ($status === 'ok') {
          $stats['ok']++;
          $ok_rows[] = [
            'submission_id' => (int)$match['id'],
            'trx_no'        => $trx_raw,
            'amount'        => $amount_val,
            'date'          => $date_val,
            'driver'        => $match['driver_name'] ?? '',
            'driver_id'     => $match['partner_id'] ?? '',
            'service_date'  => $match['service_date'] ?? '',
          ];
        } else {
          $stats['bad']++;
        }

        $preview_rows[] = [
          'row'   => $rownum,
          'trx'   => $trx_raw,
          'amount'=> $amount_raw,
          'date'  => $date_raw,
          'status'=> $status,
          'errors'=> $errors,
          'match' => $match
        ];
      }
      fclose($fh);
    }
  }
}

$payload_json = $ok_rows ? json_encode($ok_rows) : '';
$payload_sig  = $payload_json ? hmac_sign($payload_json, $csrf) : '';
?>
<!doctype html>
<html lang="en">
<head>
  <meta charset="utf-8">
  <title>Bank Statement Upload | Hisbuu</title>
  <meta name="viewport" content="width=device-width, initial-scale=1">
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
  <style>
    .row-ok { background: #f5fffa; }
    .row-err { background: #fff5f5; }
    .smallmuted { font-size: .9rem; color: #6c757d; }
    .mono { font-family: ui-monospace, SFMono-Regular, Menlo, Consolas, "Liberation Mono", monospace; }
  </style>
</head>
<body class="bg-light">
<?php include __DIR__ . '/../../includes/admin_nav.php'; ?>

<div class="container py-4" style="max-width: 1100px;">
  <div class="d-flex justify-content-between align-items-center mb-3">
    <h5 class="mb-0">Bank Statement Upload</h5>
    <div class="d-flex gap-2">
      <a href="<?php echo base_url('dashboard.php'); ?>" class="btn btn-outline-secondary btn-sm">Back</a>
      <a href="<?php echo base_url('admin/bank_upload.php'); ?>?action=template" class="btn btn-outline-secondary btn-sm">Download template</a>
    </div>
  </div>

  <?php if ($flash_ok): ?><div class="alert alert-success"><?php echo e($flash_ok); ?></div><?php endif; ?>
  <?php if ($flash_err): ?><div class="alert alert-danger"><?php echo e($flash_err); ?></div><?php endif; ?>

  <div class="card mb-4 shadow-sm">
    <div class="card-body">
      <form method="POST" enctype="multipart/form-data" class="row g-3">
        <input type="hidden" name="csrf" value="<?php echo e($csrf); ?>">
        <input type="hidden" name="phase" value="preview">
        <div class="col-md-6">
          <label class="form-label">Bank CSV file</label>
          <input type="file" name="bank_csv" class="form-control" accept=".csv" required>
          <div class="form-text">Columns: trx_no, amount, date (YYYY-MM-DD). XLSX files are not supported.</div>
        </div>
        <div class="col-md-3 d-flex align-items-end">
          <div class="form-check">
            <input class="form-check-input" type="checkbox" id="has_header" name="has_header" <?php echo $has_header ? 'checked' : ''; ?>>
            <label class="form-check-label" for="has_header">Has header row</label>
          </div>
        </div>
        <div class="col-md-3 d-flex align-items-end justify-content-end">
          <button class="btn btn-danger">Preview CSV</button>
        </div>
      </form>
    </div>
  </div>

  <?php if ($preview_rows): ?>
    <div class="card shadow-sm mb-4">
      <div class="card-header bg-white">
        <div class="d-flex justify-content-between">
          <div>
            Preview results · total <?php echo $stats['total']; ?> rows,
            <span class="text-success"><?php echo $stats['ok']; ?> ok</span>,
            <span class="text-danger"><?php echo $stats['bad']; ?> errors</span>
          </div>
        </div>
      </div>
      <div class="card-body">
        <div class="table-responsive">
          <table class="table table-sm align-middle">
            <thead>
              <tr>
                <th>#</th>
                <th>trx_no</th>
                <th class="text-end">Amount</th>
                <th>Date</th>
                <th>Matched submission</th>
                <th>Status</th>
              </tr>
            </thead>
            <tbody>
              <?php foreach ($preview_rows as $row): ?>
                <tr class="<?php echo $row['status']==='ok'?'row-ok':'row-err'; ?>">
                  <td><?php echo (int)$row['row']; ?></td>
                  <td class="mono"><?php echo e($row['trx']); ?></td>
                  <td class="text-end"><?php echo e($row['amount']); ?></td>
                  <td><?php echo e($row['date']); ?></td>
                  <td>
                    <?php if ($row['match']): ?>
                      <div><strong><?php echo e(trim($row['match']['driver_name'] ?? '')); ?></strong></div>
                      <div class="smallmuted">
                        ID <?php echo e($row['match']['partner_id'] ?? ''); ?> · Date <?php echo e($row['match']['service_date'] ?? ''); ?>
                      </div>
                    <?php else: ?>
                      <span class="text-muted">—</span>
                    <?php endif; ?>
                  </td>
                  <td>
                    <?php if ($row['status'] === 'ok'): ?>
                      <span class="badge text-bg-success">ok</span>
                    <?php else: ?>
                      <span class="badge text-bg-danger">error</span>
                      <div class="smallmuted"><?php echo e(implode('; ', $row['errors'])); ?></div>
                    <?php endif; ?>
                  </td>
                </tr>
              <?php endforeach; ?>
            </tbody>
          </table>
        </div>

        <?php if ($ok_rows): ?>
          <hr>
          <form method="POST" class="text-end">
            <input type="hidden" name="csrf" value="<?php echo e($csrf); ?>">
            <input type="hidden" name="phase" value="import">
            <input type="hidden" name="payload" value="<?php echo e($payload_json); ?>">
            <input type="hidden" name="sig" value="<?php echo e($payload_sig); ?>">
            <button class="btn btn-success">Import <?php echo count($ok_rows); ?> matched rows</button>
          </form>
        <?php endif; ?>
      </div>
    </div>
  <?php endif; ?>

  <div class="card shadow-sm">
    <div class="card-header">CSV Format Example</div>
    <div class="card-body">
      <div class="table-responsive">
        <table class="table table-bordered table-sm mb-0">
          <thead class="table-light">
            <tr><th>trx_no</th><th>amount</th><th>date</th></tr>
          </thead>
          <tbody>
            <tr><td class="mono">LFT200300400</td><td>22.356</td><td>2025-10-24</td></tr>
            <tr><td class="mono">LFT200300401</td><td>18.000</td><td>2025-10-24</td></tr>
          </tbody>
        </table>
      </div>
    </div>
  </div>
</div>

<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>
