<?php
require_once __DIR__.'/../../includes/admin_bootstrap.php';
if (!can_manage_drivers(get_current_role())) { http_response_code(403); exit('Forbidden'); }
$csrf = get_csrf_token();

$driver_id = isset($_GET['id']) && ctype_digit($_GET['id']) ? (int)$_GET['id'] : 0;
if ($driver_id <= 0) { http_response_code(400); exit('Invalid driver id'); }

// Fetch driver info
$sql = "
  SELECT
    d.id, d.partner_id, d.first_name, d.last_name,
    d.project_id, d.city_id, d.zone_id, d.nationality_id, d.type_id, d.salary_scheme_id,
    d.work_phone, d.personal_phone, d.ref_note, d.partner_uuid,
    d.status, d.created_at, d.updated_at,
    p.name  AS project_name, p.prefix AS project_prefix,
    c.name  AS city_name,
    z.name  AS zone_name,
    n.name  AS nationality_name,
    t.name  AS type_name,
    s.name  AS salary_scheme_name
  FROM drivers d
  LEFT JOIN projects p        ON p.id = d.project_id
  LEFT JOIN cities   c        ON c.id = d.city_id
  LEFT JOIN zones    z        ON z.id = d.zone_id
  LEFT JOIN nationalities n   ON n.id = d.nationality_id
  LEFT JOIN types t           ON t.id = d.type_id
  LEFT JOIN salary_schemes s  ON s.id = d.salary_scheme_id
  WHERE d.id = ?
  LIMIT 1
";
$st = $conn->prepare($sql);
$st->bind_param('i', $driver_id);
$st->execute();
$driver = $st->get_result()->fetch_assoc();
if (!$driver) { http_response_code(404); exit('Driver not found'); }

$partner_id = $driver['partner_id'];

// Get document stats
$doc_stats = $conn->query("
  SELECT
    COUNT(*) as total,
    SUM(CASE WHEN status='approved' THEN 1 ELSE 0 END) as approved,
    SUM(CASE WHEN status='pending' THEN 1 ELSE 0 END) as pending,
    SUM(CASE WHEN status='rejected' THEN 1 ELSE 0 END) as rejected
  FROM driver_doc_sets
  WHERE driver_id = $driver_id
")->fetch_assoc();

// Get submission stats
$submission_stats = $conn->query("
  SELECT
    COUNT(*) as total,
    SUM(CASE WHEN status='approved' THEN 1 ELSE 0 END) as approved,
    SUM(CASE WHEN status='pending' THEN 1 ELSE 0 END) as pending,
    SUM(CASE WHEN status='rejected' THEN 1 ELSE 0 END) as rejected,
    SUM(orders_count) as total_orders,
    SUM(cash_deposited) as total_cash
  FROM submissions
  WHERE driver_id = $driver_id
")->fetch_assoc();

// Get recent activity (last 5 days of submissions)
$recent_submissions = $conn->query("
  SELECT service_date, orders_count, cash_deposited, status, created_at, remarks
  FROM submissions
  WHERE driver_id = $driver_id
  ORDER BY service_date DESC
  LIMIT 5
")->fetch_all(MYSQLI_ASSOC);

// Get documents
$documents = $conn->query("
  SELECT
    ds.id, ds.group_key, ds.doc_number, ds.expiry_date, ds.status, ds.updated_at,
    ds.remarks,
    GROUP_CONCAT(CONCAT(dd.side, ':', dd.file_path) SEPARATOR '|') as files
  FROM driver_doc_sets ds
  LEFT JOIN driver_documents dd ON dd.doc_set_id = ds.id
  WHERE ds.driver_id = $driver_id
  GROUP BY ds.id
  ORDER BY ds.updated_at DESC
")->fetch_all(MYSQLI_ASSOC);

// Helper functions
function badge_status($s) {
  $s = strtolower((string)$s);
  if ($s === 'active')   return '<span class="badge bg-success">Active</span>';
  if ($s === 'inactive') return '<span class="badge bg-secondary">Inactive</span>';
  return '<span class="badge bg-light text-dark">'.e($s).'</span>';
}

function badge_doc_status($s) {
  $s = strtolower((string)$s);
  if ($s === 'approved') return '<span class="badge bg-success">Approved</span>';
  if ($s === 'rejected') return '<span class="badge bg-danger">Rejected</span>';
  if ($s === 'pending')  return '<span class="badge bg-warning text-dark">Pending</span>';
  return '<span class="badge bg-secondary">'.e($s).'</span>';
}

function phone_html($num) {
  if (!$num) return '-';
  $n = preg_replace('/\D+/', '', $num);
  $tel = htmlspecialchars($num, ENT_QUOTES);
  return '
    <span class="d-inline-flex align-items-center gap-2">
      <a href="tel:'.$tel.'" class="text-decoration-none text-dark"><i class="bi bi-telephone-fill"></i></a>
      <a href="https://wa.me/'.$n.'" target="_blank" class="text-success text-decoration-none"><i class="bi bi-whatsapp"></i></a>
      <span>'.e($num).'</span>
    </span>
  ';
}

function doc_type_label($key) {
  $labels = [
    'civil_id' => 'Civil ID Card',
    'license' => 'Driving License',
    'vehicle_reg' => 'Vehicle Registration'
  ];
  return $labels[$key] ?? ucwords(str_replace('_', ' ', $key));
}

$PAGE_TITLE = 'Driver: ' . trim($driver['first_name'] . ' ' . $driver['last_name']);
require_once __DIR__ . '/../../includes/admin_header.php';
?>

<link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.3/font/bootstrap-icons.css" rel="stylesheet">

<style>
.stat-card {
  border-left: 4px solid #dee2e6;
  transition: all 0.2s;
}
.stat-card:hover {
  transform: translateY(-2px);
  box-shadow: 0 4px 12px rgba(0,0,0,0.1);
}
.stat-card.stat-primary { border-left-color: #dc3545; }
.stat-card.stat-success { border-left-color: #198754; }
.stat-card.stat-warning { border-left-color: #ffc107; }
.stat-card.stat-danger { border-left-color: #dc3545; }
.stat-card.stat-info { border-left-color: #0dcaf0; }

.nav-pills .nav-link {
  color: #6c757d;
  border-radius: 0.5rem;
}
.nav-pills .nav-link.active {
  background-color: #dc3545;
}
.nav-pills .nav-link:hover:not(.active) {
  background-color: #e9ecef;
}

.driver-avatar {
  width: 80px;
  height: 80px;
  border-radius: 50%;
  background: linear-gradient(135deg, #dc3545 0%, #c82333 100%);
  display: flex;
  align-items: center;
  justify-content: center;
  color: white;
  font-size: 2rem;
  font-weight: 600;
}

.info-row {
  padding: 0.75rem 0;
  border-bottom: 1px solid #f0f0f0;
}
.info-row:last-child {
  border-bottom: none;
}
.info-label {
  font-weight: 600;
  color: #6c757d;
  min-width: 200px;
}

.timeline-item {
  position: relative;
  padding-left: 30px;
  padding-bottom: 1.5rem;
}
.timeline-item:before {
  content: '';
  position: absolute;
  left: 7px;
  top: 8px;
  bottom: -8px;
  width: 2px;
  background: #e9ecef;
}
.timeline-item:last-child:before {
  display: none;
}
.timeline-dot {
  position: absolute;
  left: 0;
  top: 8px;
  width: 16px;
  height: 16px;
  border-radius: 50%;
  background: white;
  border: 3px solid #dc3545;
  z-index: 1;
}
</style>

<div class="d-flex justify-content-between align-items-start mb-4">
  <div>
    <h4 class="mb-1">Driver Profile</h4>
    <p class="text-muted small mb-0">Comprehensive view of driver information and activity</p>
  </div>
  <div class="d-flex gap-2">
    <button type="button" id="toggleStatusBtn" class="btn btn-sm <?= strtolower($driver['status']) === 'active' ? 'btn-warning' : 'btn-success' ?>" data-driver-id="<?= $driver_id ?>" data-current-status="<?= strtolower($driver['status']) ?>">
      <i class="bi bi-<?= strtolower($driver['status']) === 'active' ? 'pause-circle' : 'play-circle' ?>"></i>
      <?= strtolower($driver['status']) === 'active' ? 'Deactivate' : 'Activate' ?>
    </button>
    <a href="<?= base_url('admin/driver_list.php') ?>" class="btn btn-outline-secondary btn-sm">
      <i class="bi bi-arrow-left"></i> Back to List
    </a>
    <a href="<?= base_url('admin/driver_edit.php?id='.$driver['id']) ?>" class="btn btn-primary btn-sm">
      <i class="bi bi-pencil"></i> Edit Driver
    </a>
  </div>
</div>

<!-- Driver Header Card -->
<div class="card shadow-sm mb-4">
  <div class="card-body">
    <div class="row align-items-center">
      <div class="col-auto">
        <div class="driver-avatar">
          <?= strtoupper(substr($driver['first_name'] ?? 'D', 0, 1)) ?>
        </div>
      </div>
      <div class="col">
        <h5 class="mb-1"><?= e(trim($driver['first_name'].' '.$driver['last_name'])) ?></h5>
        <div class="text-muted mb-2">
          <span class="me-3"><i class="bi bi-person-badge me-1"></i><?= e($driver['partner_id']) ?></span>
          <span class="me-3"><i class="bi bi-building me-1"></i><?= e($driver['project_name'] ?? '-') ?></span>
          <span id="status-badge"><?= badge_status($driver['status']) ?></span>
        </div>
        <div class="d-flex gap-3 small text-muted">
          <span><i class="bi bi-calendar-plus me-1"></i>Joined <?= date('M d, Y', strtotime($driver['created_at'])) ?></span>
          <span><i class="bi bi-clock-history me-1"></i>Updated <?= date('M d, Y', strtotime($driver['updated_at'])) ?></span>
        </div>
      </div>
    </div>
  </div>
</div>

<!-- Stats Cards -->
<div class="row g-3 mb-4">
  <div class="col-md-3">
    <div class="card stat-card stat-primary shadow-sm h-100">
      <div class="card-body">
        <div class="d-flex justify-content-between align-items-start">
          <div>
            <div class="text-muted small mb-1">Total Submissions</div>
            <h3 class="mb-0"><?= number_format($submission_stats['total'] ?? 0) ?></h3>
          </div>
          <div class="bg-primary bg-opacity-10 p-3 rounded">
            <i class="bi bi-file-text fs-4 text-primary"></i>
          </div>
        </div>
      </div>
    </div>
  </div>

  <div class="col-md-3">
    <div class="card stat-card stat-success shadow-sm h-100">
      <div class="card-body">
        <div class="d-flex justify-content-between align-items-start">
          <div>
            <div class="text-muted small mb-1">Total Orders</div>
            <h3 class="mb-0"><?= number_format($submission_stats['total_orders'] ?? 0) ?></h3>
          </div>
          <div class="bg-success bg-opacity-10 p-3 rounded">
            <i class="bi bi-bag-check fs-4 text-success"></i>
          </div>
        </div>
      </div>
    </div>
  </div>

  <div class="col-md-3">
    <div class="card stat-card stat-info shadow-sm h-100">
      <div class="card-body">
        <div class="d-flex justify-content-between align-items-start">
          <div>
            <div class="text-muted small mb-1">Total Cash</div>
            <h3 class="mb-0"><?= number_format($submission_stats['total_cash'] ?? 0, 3) ?></h3>
          </div>
          <div class="bg-info bg-opacity-10 p-3 rounded">
            <i class="bi bi-cash-stack fs-4 text-info"></i>
          </div>
        </div>
      </div>
    </div>
  </div>

  <div class="col-md-3">
    <div class="card stat-card stat-warning shadow-sm h-100">
      <div class="card-body">
        <div class="d-flex justify-content-between align-items-start">
          <div>
            <div class="text-muted small mb-1">Documents</div>
            <h3 class="mb-0"><?= ($doc_stats['approved'] ?? 0) ?>/<?= ($doc_stats['total'] ?? 0) ?></h3>
            <small class="text-muted">Approved</small>
          </div>
          <div class="bg-warning bg-opacity-10 p-3 rounded">
            <i class="bi bi-file-earmark-check fs-4 text-warning"></i>
          </div>
        </div>
      </div>
    </div>
  </div>
</div>

<!-- Tabs Navigation -->
<ul class="nav nav-pills mb-4" id="driverTabs" role="tablist">
  <li class="nav-item" role="presentation">
    <button class="nav-link active" id="info-tab" data-bs-toggle="pill" data-bs-target="#info" type="button">
      <i class="bi bi-person-circle me-2"></i>Basic Info
    </button>
  </li>
  <li class="nav-item" role="presentation">
    <button class="nav-link" id="documents-tab" data-bs-toggle="pill" data-bs-target="#documents" type="button">
      <i class="bi bi-file-earmark-text me-2"></i>Documents (<?= $doc_stats['total'] ?? 0 ?>)
    </button>
  </li>
  <li class="nav-item" role="presentation">
    <button class="nav-link" id="activity-tab" data-bs-toggle="pill" data-bs-target="#activity" type="button">
      <i class="bi bi-clock-history me-2"></i>Recent Activity
    </button>
  </li>
  <li class="nav-item" role="presentation">
    <button class="nav-link" id="submissions-tab" data-bs-toggle="pill" data-bs-target="#submissions" type="button">
      <i class="bi bi-list-check me-2"></i>All Submissions
    </button>
  </li>
</ul>

<!-- Tab Content -->
<div class="tab-content" id="driverTabsContent">

  <!-- Basic Info Tab -->
  <div class="tab-pane fade show active" id="info" role="tabpanel">
    <div class="card shadow-sm">
      <div class="card-header bg-white">
        <h6 class="mb-0">Driver Information</h6>
      </div>
      <div class="card-body">
        <div class="row">
          <div class="col-md-6">
            <div class="info-row d-flex">
              <div class="info-label">Driver ID</div>
              <div class="flex-grow-1 fw-semibold"><?= e($driver['partner_id']) ?></div>
            </div>
            <div class="info-row d-flex">
              <div class="info-label">Partner UUID</div>
              <div class="flex-grow-1"><code class="small"><?= e($driver['partner_uuid'] ?? '-') ?></code></div>
            </div>
            <div class="info-row d-flex">
              <div class="info-label">Full Name</div>
              <div class="flex-grow-1"><?= e(trim($driver['first_name'].' '.$driver['last_name'])) ?></div>
            </div>
            <div class="info-row d-flex">
              <div class="info-label">Project</div>
              <div class="flex-grow-1">
                <?= e($driver['project_name'] ?? '-') ?>
                <?php if (!empty($driver['project_prefix'])): ?>
                  <span class="badge bg-light text-dark ms-2">Prefix: <?= e($driver['project_prefix']) ?></span>
                <?php endif; ?>
              </div>
            </div>
            <div class="info-row d-flex">
              <div class="info-label">City</div>
              <div class="flex-grow-1"><?= e($driver['city_name'] ?? '-') ?></div>
            </div>
            <div class="info-row d-flex">
              <div class="info-label">Zone</div>
              <div class="flex-grow-1"><?= e($driver['zone_name'] ?? '-') ?></div>
            </div>
          </div>

          <div class="col-md-6">
            <div class="info-row d-flex">
              <div class="info-label">Nationality</div>
              <div class="flex-grow-1"><?= e($driver['nationality_name'] ?? '-') ?></div>
            </div>
            <div class="info-row d-flex">
              <div class="info-label">Type</div>
              <div class="flex-grow-1"><?= e($driver['type_name'] ?? '-') ?></div>
            </div>
            <div class="info-row d-flex">
              <div class="info-label">Salary Scheme</div>
              <div class="flex-grow-1"><?= e($driver['salary_scheme_name'] ?? '-') ?></div>
            </div>
            <div class="info-row d-flex">
              <div class="info-label">Work Phone</div>
              <div class="flex-grow-1"><?= phone_html($driver['work_phone'] ?? '') ?></div>
            </div>
            <div class="info-row d-flex">
              <div class="info-label">Personal Phone</div>
              <div class="flex-grow-1"><?= phone_html($driver['personal_phone'] ?? '') ?></div>
            </div>
            <div class="info-row d-flex">
              <div class="info-label">Status</div>
              <div class="flex-grow-1"><?= badge_status($driver['status']) ?></div>
            </div>
          </div>
        </div>

        <?php if (!empty($driver['ref_note'])): ?>
        <div class="mt-3 p-3 bg-light rounded">
          <div class="small text-muted mb-1">Reference Note</div>
          <div><?= e($driver['ref_note']) ?></div>
        </div>
        <?php endif; ?>
      </div>
    </div>
  </div>

  <!-- Documents Tab -->
  <div class="tab-pane fade" id="documents" role="tabpanel">
    <?php if (empty($documents)): ?>
      <div class="card shadow-sm">
        <div class="card-body text-center py-5">
          <i class="bi bi-file-earmark-x fs-1 text-muted"></i>
          <p class="text-muted mt-3">No documents uploaded yet</p>
          <a href="<?= base_url('admin/driver_doc_sets_review.php?driver='.urlencode($partner_id)) ?>" class="btn btn-primary btn-sm">
            View KYD Management
          </a>
        </div>
      </div>
    <?php else: ?>
      <div class="row row-cols-1 row-cols-md-2 row-cols-lg-3 g-3">
        <?php foreach ($documents as $doc):
          $files = [];
          if (!empty($doc['files'])) {
            foreach (explode('|', $doc['files']) as $f) {
              if (strpos($f, ':') !== false) {
                list($side, $path) = explode(':', $f, 2);
                $files[$side] = $path;
              }
            }
          }
        ?>
        <div class="col">
          <div class="card shadow-sm h-100">
            <div class="card-header bg-white d-flex justify-content-between align-items-center">
              <h6 class="mb-0"><?= doc_type_label($doc['group_key']) ?></h6>
              <?= badge_doc_status($doc['status']) ?>
            </div>
            <div class="card-body">
              <?php if ($doc['doc_number']): ?>
                <div class="small mb-2">
                  <strong>Number:</strong> <?= e($doc['doc_number']) ?>
                </div>
              <?php endif; ?>
              <?php if ($doc['expiry_date']): ?>
                <div class="small mb-3">
                  <strong>Expiry:</strong> <?= date('M d, Y', strtotime($doc['expiry_date'])) ?>
                </div>
              <?php endif; ?>

              <div class="row g-2">
                <?php foreach (['front', 'back'] as $side): ?>
                  <div class="col-6">
                    <div class="text-center border rounded p-2 bg-light">
                      <div class="small text-muted mb-1"><?= ucfirst($side) ?></div>
                      <?php if (isset($files[$side])): ?>
                        <a href="<?= file_url($files[$side]) ?>" target="_blank">
                          <img src="<?= file_url($files[$side]) ?>" class="img-fluid rounded" style="max-height: 80px; object-fit: cover;">
                        </a>
                      <?php else: ?>
                        <div class="text-muted small py-3">
                          <i class="bi bi-image opacity-25"></i>
                        </div>
                      <?php endif; ?>
                    </div>
                  </div>
                <?php endforeach; ?>
              </div>

              <?php if ($doc['remarks']): ?>
                <div class="alert alert-warning py-2 mt-3 mb-0 small">
                  <strong>Note:</strong> <?= e($doc['remarks']) ?>
                </div>
              <?php endif; ?>

              <div class="text-muted small mt-2">
                Updated: <?= date('M d, Y', strtotime($doc['updated_at'])) ?>
              </div>
            </div>
          </div>
        </div>
        <?php endforeach; ?>
      </div>

      <div class="text-center mt-4">
        <a href="<?= base_url('admin/driver_doc_sets_review.php?driver='.urlencode($partner_id)) ?>" class="btn btn-outline-primary">
          View All Documents & Manage <i class="bi bi-arrow-right ms-2"></i>
        </a>
      </div>
    <?php endif; ?>
  </div>

  <!-- Recent Activity Tab -->
  <div class="tab-pane fade" id="activity" role="tabpanel">
    <div class="card shadow-sm">
      <div class="card-header bg-white">
        <h6 class="mb-0">Last 5 Days Activity</h6>
      </div>
      <div class="card-body">
        <?php if (empty($recent_submissions)): ?>
          <div class="text-center py-5">
            <i class="bi bi-calendar-x fs-1 text-muted"></i>
            <p class="text-muted mt-3">No recent activity</p>
          </div>
        <?php else: ?>
          <div class="timeline">
            <?php foreach ($recent_submissions as $sub): ?>
              <div class="timeline-item">
                <div class="timeline-dot"></div>
                <div class="card border-0 bg-light">
                  <div class="card-body p-3">
                    <div class="d-flex justify-content-between align-items-start mb-2">
                      <div>
                        <strong><?= date('l, M d, Y', strtotime($sub['service_date'])) ?></strong>
                        <div class="small text-muted">Submitted <?= date('M d, Y g:i A', strtotime($sub['created_at'])) ?></div>
                      </div>
                      <?= badge_doc_status($sub['status']) ?>
                    </div>
                    <div class="row g-2">
                      <div class="col-auto">
                        <span class="badge bg-primary bg-opacity-10 text-primary">
                          <i class="bi bi-bag-check me-1"></i><?= $sub['orders_count'] ?> orders
                        </span>
                      </div>
                      <div class="col-auto">
                        <span class="badge bg-success bg-opacity-10 text-success">
                          <i class="bi bi-cash me-1"></i><?= number_format($sub['cash_deposited'], 3) ?> KD
                        </span>
                      </div>
                    </div>
                    <?php if ($sub['remarks']): ?>
                      <div class="mt-2 small text-muted">
                        <i class="bi bi-chat-left-text me-1"></i><?= e($sub['remarks']) ?>
                      </div>
                    <?php endif; ?>
                  </div>
                </div>
              </div>
            <?php endforeach; ?>
          </div>
        <?php endif; ?>
      </div>
    </div>
  </div>

  <!-- All Submissions Tab -->
  <div class="tab-pane fade" id="submissions" role="tabpanel">
    <div class="card shadow-sm">
      <div class="card-header bg-white d-flex justify-content-between align-items-center">
        <h6 class="mb-0">Submission History</h6>
        <a href="<?= base_url('admin/submissions_list.php?driver='.urlencode($partner_id)) ?>" class="btn btn-sm btn-outline-primary">
          View Full List <i class="bi bi-arrow-right ms-1"></i>
        </a>
      </div>
      <div class="card-body">
        <div class="row g-3 mb-3">
          <div class="col-md-3">
            <div class="text-center p-3 bg-light rounded">
              <div class="text-muted small">Total</div>
              <div class="fs-4 fw-bold"><?= number_format($submission_stats['total'] ?? 0) ?></div>
            </div>
          </div>
          <div class="col-md-3">
            <div class="text-center p-3 bg-success bg-opacity-10 rounded">
              <div class="text-muted small">Approved</div>
              <div class="fs-4 fw-bold text-success"><?= number_format($submission_stats['approved'] ?? 0) ?></div>
            </div>
          </div>
          <div class="col-md-3">
            <div class="text-center p-3 bg-warning bg-opacity-10 rounded">
              <div class="text-muted small">Pending</div>
              <div class="fs-4 fw-bold text-warning"><?= number_format($submission_stats['pending'] ?? 0) ?></div>
            </div>
          </div>
          <div class="col-md-3">
            <div class="text-center p-3 bg-danger bg-opacity-10 rounded">
              <div class="text-muted small">Rejected</div>
              <div class="fs-4 fw-bold text-danger"><?= number_format($submission_stats['rejected'] ?? 0) ?></div>
            </div>
          </div>
        </div>

        <div class="alert alert-info mb-0">
          <i class="bi bi-info-circle me-2"></i>
          Use the "View Full List" button above to see all submissions with filters and export options.
        </div>
      </div>
    </div>
  </div>
</div>

<!-- Quick Actions -->
<div class="card shadow-sm mt-4">
  <div class="card-header bg-white">
    <h6 class="mb-0"><i class="bi bi-lightning-charge me-2"></i>Quick Actions</h6>
  </div>
  <div class="card-body">
    <div class="row g-2">
      <div class="col-md-3">
        <a class="btn btn-outline-primary btn-sm w-100" href="<?= base_url('admin/pending_approvals.php?driver='.urlencode($partner_id)) ?>">
          <i class="bi bi-clock-history me-2"></i>Pending Approvals
        </a>
      </div>
      <div class="col-md-3">
        <a class="btn btn-outline-primary btn-sm w-100" href="<?= base_url('admin/submissions_list.php?driver='.urlencode($partner_id)) ?>">
          <i class="bi bi-list-check me-2"></i>All Submissions
        </a>
      </div>
      <div class="col-md-3">
        <a class="btn btn-outline-primary btn-sm w-100" href="<?= base_url('admin/driver_doc_sets_review.php?driver='.urlencode($partner_id)) ?>">
          <i class="bi bi-file-earmark-check me-2"></i>KYD Documents
        </a>
      </div>
      <div class="col-md-3">
        <a class="btn btn-outline-primary btn-sm w-100" href="<?= base_url('admin/adjustments_list.php?driver_id='.$driver['id']) ?>">
          <i class="bi bi-calculator me-2"></i>Adjustments
        </a>
      </div>
    </div>
  </div>
</div>

<script src="https://code.jquery.com/jquery-3.7.1.min.js"></script>
<script>
$(function() {
  // Toggle driver status
  $('#toggleStatusBtn').on('click', function() {
    const $btn = $(this);
    const driverId = $btn.data('driver-id');
    const currentStatus = $btn.data('current-status');
    const newStatus = currentStatus === 'active' ? 'inactive' : 'active';

    // Confirmation dialog
    const confirmMsg = newStatus === 'active'
      ? 'Are you sure you want to ACTIVATE this driver?\n\nThe driver will be able to log in and submit data.'
      : 'Are you sure you want to DEACTIVATE this driver?\n\nThe driver will be logged out and unable to access the system.';

    if (!confirm(confirmMsg)) {
      return;
    }

    // Disable button during request
    const originalHtml = $btn.html();
    $btn.prop('disabled', true).html('<i class="bi bi-hourglass-split"></i> Processing...');

    $.ajax({
      url: '<?= base_url('admin/driver_toggle_status.php') ?>',
      method: 'POST',
      data: {
        driver_id: driverId,
        new_status: newStatus,
        csrf: '<?= $csrf ?>'
      },
      dataType: 'json',
      success: function(response) {
        if (response.success) {
          // Update button state
          $btn.data('current-status', newStatus);

          if (newStatus === 'active') {
            $btn.removeClass('btn-success').addClass('btn-warning');
            $btn.html('<i class="bi bi-pause-circle"></i> Deactivate');
          } else {
            $btn.removeClass('btn-warning').addClass('btn-success');
            $btn.html('<i class="bi bi-play-circle"></i> Activate');
          }

          // Update status badge
          const badgeClass = newStatus === 'active' ? 'bg-success' : 'bg-secondary';
          $('#status-badge').html('<span class="badge ' + badgeClass + '">' +
            (newStatus.charAt(0).toUpperCase() + newStatus.slice(1)) + '</span>');

          // Show success notification
          const alertHtml = '<div class="alert alert-success alert-dismissible fade show mt-3" role="alert">' +
            '<i class="bi bi-check-circle-fill me-2"></i>' +
            'Driver has been successfully ' + (newStatus === 'active' ? 'activated' : 'deactivated') + '.' +
            '<button type="button" class="btn-close" data-bs-dismiss="alert"></button>' +
            '</div>';

          $('.mb-4:first').after(alertHtml);

          // Auto-dismiss after 5 seconds
          setTimeout(function() {
            $('.alert-success').alert('close');
          }, 5000);

          $btn.prop('disabled', false);
        } else {
          alert('Error: ' + (response.error || 'Failed to update driver status'));
          $btn.prop('disabled', false).html(originalHtml);
        }
      },
      error: function(xhr) {
        alert('An error occurred while updating the driver status. Please try again.');
        $btn.prop('disabled', false).html(originalHtml);
      }
    });
  });
});
</script>

<?php require_once __DIR__ . '/../../includes/admin_footer.php'; ?>
