<?php
// public/admin/login.php
require_once __DIR__ . '/../../includes/helpers.php';
require_once __DIR__ . '/../../includes/auth_check.php';
require_once __DIR__ . '/../../includes/settings.php';
require_once __DIR__ . '/../../config/db.php';

redirect_if_logged_in();                 // stops loop from login side
if (session_status() === PHP_SESSION_NONE) session_start();

// Fetch branding settings
$appName = app_setting('branding.app_name', 'Hisbuu');
$loginLogo = app_setting('branding.admin_login_logo', '');
if ($loginLogo !== '' && strpos($loginLogo, 'storage/') === 0) {
    $loginLogo = file_url($loginLogo);
}

if (empty($_SESSION['csrf'])) $_SESSION['csrf'] = bin2hex(random_bytes(16));
$csrf = $_SESSION['csrf'];
$err = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['csrf']) && hash_equals($csrf, $_POST['csrf'])) {
    $email = trim($_POST['email'] ?? '');
    $pass  = (string)($_POST['password'] ?? '');

    if ($email === '' || $pass === '') {
        $err = 'Email and password are required.';
    } else {
        // Verify credentials
        $stmt = $conn->prepare("SELECT id,email,password_hash,role,status FROM users WHERE email=? LIMIT 1");
        $stmt->bind_param('s', $email);
        $stmt->execute();
        $u = $stmt->get_result()->fetch_assoc();

        if (!$u || (isset($u['status']) && $u['status'] === 'disabled')) {
            $err = 'Invalid credentials.';
        } elseif (!password_verify($pass, $u['password_hash'])) {
            $err = 'Invalid credentials.';
        } else {
            $_SESSION['user'] = [
                'id'    => (int)$u['id'],
                'email' => $u['email'],
                'role'  => $u['role'] ?? 'Administrator',
            ];
            session_regenerate_id(true);
            header('Location: ../dashboard.php');
            exit;
        }
    }
}
?>
<!doctype html>
<html lang="en">
<head>
  <meta charset="utf-8">
  <title>Admin Login | <?php echo e($appName); ?></title>
  <meta name="viewport" content="width=device-width, initial-scale=1">
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
  <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.3/font/bootstrap-icons.min.css" rel="stylesheet">
  <style>
    :root {
      --red: #dc3545;
      --red-dark: #b02a37;
      --red-soft: #fee2e2;
      --gray-50: #f8fafc;
      --gray-100: #f1f5f9;
      --gray-400: #94a3b8;
      --gray-500: #64748b;
      --gray-600: #475569;
      --gray-900: #0f172a;
    }

    body {
      background: var(--gray-50);
      font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
      min-height: 100vh;
      display: flex;
      align-items: center;
      padding: 20px;
    }

    .login-wrapper {
      width: 100%;
      max-width: 1100px;
      margin: 0 auto;
    }

    /* Left Hero Section */
    .hero-section {
      background: linear-gradient(135deg, var(--red) 0%, var(--red-dark) 100%);
      border-radius: 20px;
      padding: 48px 40px;
      color: #fff;
      height: 100%;
      display: flex;
      flex-direction: column;
      justify-content: center;
    }

    .logo-wrapper {
      margin-bottom: 24px;
    }

    .logo-wrapper img {
      max-height: 40px;
      max-width: 180px;
      object-fit: contain;
      filter: brightness(0) invert(1);
    }

    .brand-badge {
      background: rgba(255,255,255,0.2);
      backdrop-filter: blur(10px);
      color: #fff;
      border-radius: 12px;
      padding: 8px 16px;
      font-weight: 600;
      font-size: 16px;
      display: inline-block;
    }

    .hero-section h2 {
      font-size: 32px;
      font-weight: 700;
      margin-bottom: 24px;
      line-height: 1.3;
    }

    .feature-list {
      list-style: none;
      padding: 0;
      margin: 0 0 24px;
    }

    .feature-list li {
      display: flex;
      align-items: center;
      gap: 12px;
      padding: 10px 0;
      font-size: 15px;
      opacity: 0.95;
    }

    .feature-list li i {
      font-size: 18px;
      opacity: 0.8;
    }

    .hero-footer {
      font-size: 13px;
      opacity: 0.75;
      margin-top: auto;
      padding-top: 24px;
      border-top: 1px solid rgba(255,255,255,0.15);
    }

    /* Right Login Section */
    .login-section {
      background: #fff;
      border-radius: 20px;
      padding: 48px 40px;
      box-shadow: 0 4px 6px rgba(0,0,0,0.02), 0 10px 20px rgba(0,0,0,0.04);
    }

    .login-header {
      margin-bottom: 32px;
    }

    .login-header h3 {
      font-size: 24px;
      font-weight: 700;
      color: var(--gray-900);
      margin-bottom: 6px;
    }

    .login-header p {
      font-size: 14px;
      color: var(--gray-500);
      margin: 0;
    }

    .form-label {
      font-size: 14px;
      font-weight: 500;
      color: var(--gray-600);
      margin-bottom: 8px;
    }

    .form-control {
      border: 1.5px solid #e2e8f0;
      border-radius: 12px;
      padding: 12px 16px;
      font-size: 15px;
      transition: border-color 0.15s, box-shadow 0.15s;
    }

    .form-control:focus {
      border-color: var(--red);
      box-shadow: 0 0 0 3px var(--red-soft);
    }

    .input-group-icon {
      position: relative;
    }

    .input-group-icon i {
      position: absolute;
      left: 16px;
      top: 50%;
      transform: translateY(-50%);
      color: var(--gray-400);
      font-size: 18px;
      pointer-events: none;
    }

    .input-group-icon .form-control {
      padding-left: 48px;
    }

    .btn-login {
      width: 100%;
      background: var(--red);
      border: none;
      border-radius: 12px;
      padding: 13px 20px;
      font-size: 16px;
      font-weight: 600;
      color: #fff;
      transition: background 0.15s, transform 0.1s;
    }

    .btn-login:hover {
      background: var(--red-dark);
    }

    .btn-login:active {
      transform: scale(0.98);
    }

    .alert-error {
      background: var(--red-soft);
      border: none;
      border-radius: 12px;
      color: var(--red-dark);
      padding: 12px 16px;
      font-size: 14px;
      display: flex;
      align-items: center;
      gap: 10px;
      margin-bottom: 24px;
    }

    .alert-error i {
      font-size: 18px;
    }

    .login-footer {
      margin-top: 24px;
      text-align: center;
    }

    .login-footer a {
      color: var(--gray-500);
      font-size: 14px;
      text-decoration: none;
      display: inline-flex;
      align-items: center;
      gap: 6px;
      transition: color 0.15s;
    }

    .login-footer a:hover {
      color: var(--red);
    }

    .login-footer a i {
      font-size: 16px;
    }

    /* Responsive */
    @media (max-width: 991px) {
      .hero-section {
        padding: 32px 28px;
        margin-bottom: 24px;
      }

      .hero-section h2 {
        font-size: 24px;
      }

      .login-section {
        padding: 32px 28px;
      }

      .login-header h3 {
        font-size: 20px;
      }
    }

    @media (max-width: 575px) {
      body {
        padding: 12px;
      }

      .hero-section,
      .login-section {
        padding: 24px 20px;
      }

      .hero-section h2 {
        font-size: 20px;
      }

      .feature-list li {
        font-size: 14px;
      }
    }
  </style>
</head>
<body>

<div class="login-wrapper">
  <div class="row g-4">
    <!-- Left: Hero -->
    <div class="col-lg-6">
      <div class="hero-section">
        <div class="logo-wrapper">
          <?php if ($loginLogo): ?>
            <img src="<?php echo e($loginLogo); ?>" alt="<?php echo e($appName); ?>">
          <?php else: ?>
            <span class="brand-badge"><?php echo e($appName); ?></span>
          <?php endif; ?>
        </div>

        <h2>Fleet and finance control, made clear</h2>

        <ul class="feature-list">
          <li><i class="bi bi-check-circle-fill"></i> Track daily orders and cash by driver</li>
          <li><i class="bi bi-check-circle-fill"></i> Approve submissions with screenshots</li>
          <li><i class="bi bi-check-circle-fill"></i> Match deposits using bank transactions</li>
          <li><i class="bi bi-check-circle-fill"></i> Filter by partner, city, and zone</li>
        </ul>

        <div class="hero-footer">
          <i class="bi bi-info-circle"></i> Drivers have a separate login portal
        </div>
      </div>
    </div>

    <!-- Right: Login -->
    <div class="col-lg-6">
      <div class="login-section">
        <div class="login-header">
          <h3>Admin Login</h3>
          <p>Sign in to access your admin dashboard</p>
        </div>

        <?php if ($err): ?>
          <div class="alert-error">
            <i class="bi bi-exclamation-circle"></i>
            <span><?php echo e($err); ?></span>
          </div>
        <?php endif; ?>

        <form method="post">
          <input type="hidden" name="csrf" value="<?php echo e($csrf); ?>">

          <div class="mb-3">
            <label class="form-label">Email Address</label>
            <div class="input-group-icon">
              <i class="bi bi-envelope"></i>
              <input type="email" name="email" class="form-control" placeholder="your.email@example.com" required autocomplete="username">
            </div>
          </div>

          <div class="mb-4">
            <label class="form-label">Password</label>
            <div class="input-group-icon">
              <i class="bi bi-lock"></i>
              <input type="password" name="password" class="form-control" placeholder="Enter your password" required autocomplete="current-password">
            </div>
          </div>

          <button type="submit" class="btn-login">
            Sign in to Dashboard
          </button>
        </form>

        <div class="login-footer">
          <a href="../driver/login.php">
            <i class="bi bi-truck"></i>
            Driver Login Portal
          </a>
        </div>
      </div>
    </div>
  </div>
</div>

</body>
</html>
