<?php
// public/admin/partner_upload.php
require_once __DIR__ . '/../../includes/admin_bootstrap.php';
$role = strtolower(get_current_role() ?? '');
if (!in_array($role, ['super administrator','administrator','finance administrator'], true)) {
  http_response_code(403); exit('Forbidden');
}

if (session_status() === PHP_SESSION_NONE) session_start();
if (empty($_SESSION['csrf'])) $_SESSION['csrf'] = bin2hex(random_bytes(16));
$csrf = $_SESSION['csrf'];

header('X-Frame-Options: SAMEORIGIN');

function download_template() {
  header('Content-Type: text/csv; charset=utf-8');
  header('Content-Disposition: attachment; filename="partner_daily_template.csv"');
  $out = fopen('php://output', 'w');
  // Keep columns simple and human: we will resolve IDs internally.
  fputcsv($out, ['date','driver_id','driver_name','total_orders','total_amount','remarks']);
  fputcsv($out, ['2025-10-27','TB11070108','Ahmed Al Balushi','26','143.350','Reconciled']);
  fputcsv($out, ['2025-10-27','KD99887766','Rizwan Zafar','18','99.000','']);
  fclose($out);
  exit;
}
if (isset($_GET['action']) && $_GET['action']==='template') download_template();

function hmac_sign(string $data, string $key): string {
  return hash_hmac('sha256', $data, $key);
}

$flash_ok=''; $flash_err='';
$is_preview = ($_SERVER['REQUEST_METHOD']==='POST' && ($_POST['phase'] ?? '')==='preview');
$is_import  = ($_SERVER['REQUEST_METHOD']==='POST' && ($_POST['phase'] ?? '')==='import');

$preview_rows = [];
$ok_rows = [];
$stats = ['total'=>0,'ok'=>0,'bad'=>0];

// Phase 2: Import confirmed rows
if ($is_import && isset($_POST['csrf']) && hash_equals($csrf, $_POST['csrf'])) {
  $payload = trim($_POST['payload'] ?? '');
  $sig     = trim($_POST['sig'] ?? '');
  if ($payload==='' || $sig==='') {
    $flash_err = 'Missing import payload.';
  } else {
    $expected = hmac_sign($payload, $csrf);
    if (!hash_equals($expected, $sig)) {
      $flash_err = 'Invalid or expired token.';
    } else {
      $ok_rows = json_decode($payload, true);
      if (!is_array($ok_rows)) {
        $flash_err = 'Payload malformed.';
      } else {
        $ins = $conn->prepare("
          INSERT INTO partner_daily
            (service_date, driver_id, project_id, total_orders, total_amount, remarks, created_at, updated_at)
          VALUES (?,?,?,?,?,?,NOW(),NOW())
        ");
        $ins->bind_param('siiids', $d, $driver_id, $project_id, $orders, $amount, $remarks);

        $inserted = 0; $skipped = 0;
        foreach ($ok_rows as $r) {
          $d         = $r['service_date'];           // Y-m-d
          $driver_id = (int)$r['driver_id'];
          $project_id= (int)$r['project_id'];        // already resolved in preview
          $orders    = (int)$r['total_orders'];
          $amount    = (float)$r['total_amount'];    // 3 decimal allowed
          $remarks   = $r['remarks'];

          // Guard: ensure uniqueness per (service_date, driver_id)
          $chk = $conn->prepare("SELECT id FROM partner_daily WHERE service_date=? AND driver_id=? LIMIT 1");
          $chk->bind_param('si', $d, $driver_id);
          $chk->execute();
          if ($chk->get_result()->fetch_assoc()) { $skipped++; continue; }

          if ($ins->execute()) $inserted++; else $skipped++;
        }
        $flash_ok = "Imported $inserted. Skipped $skipped.";
      }
    }
  }
}

// Phase 1: CSV preview and validation
if ($is_preview && isset($_POST['csrf']) && hash_equals($csrf, $_POST['csrf'])) {
  if (!isset($_FILES['csv']) || $_FILES['csv']['error']!==UPLOAD_ERR_OK) {
    $flash_err = 'CSV upload failed.';
  } else {
    $fh = fopen($_FILES['csv']['tmp_name'], 'r');
    if (!$fh) {
      $flash_err = 'Cannot open uploaded file.';
    } else {
      $has_header = !empty($_POST['has_header']);
      $rownum = 0;

      // Caches
      $cache_driver = [];  // partner_id => [id, project_id, driver_name]
      while (($cols = fgetcsv($fh)) !== false) {
        $rownum++;
        if ($has_header && $rownum===1) continue;

        $stats['total']++;
        $cols = array_pad($cols, 6, '');
        list($date, $partner_id, $driver_name_csv, $total_orders, $total_amount, $remarks) =
          array_map('trim', $cols);

        $errors = [];

        // Date check
        $d = DateTime::createFromFormat('Y-m-d', $date);
        $valid_date = $d && $d->format('Y-m-d') === $date;
        if (!$valid_date) $errors[] = 'invalid date, use YYYY-MM-DD';

        // Partner ID check
        if ($partner_id==='') {
          $errors[] = 'driver_id missing';
        } else {
          if (!isset($cache_driver[$partner_id])) {
            $st = $conn->prepare("
              SELECT d.id, d.project_id, CONCAT(d.first_name,' ',d.last_name) AS nm
              FROM drivers d WHERE d.partner_id=? LIMIT 1
            ");
            $st->bind_param('s', $partner_id);
            $st->execute();
            $row = $st->get_result()->fetch_assoc();
            $cache_driver[$partner_id] = $row ? [
              'id' => (int)$row['id'],
              'project_id' => (int)$row['project_id'],
              'name' => $row['nm'] ?? ''
            ] : null;
          }
          if (!$cache_driver[$partner_id]) {
            $errors[] = 'driver_id not found';
          }
        }

        // Orders
        if ($total_orders === '' || !ctype_digit($total_orders)) {
          $errors[] = 'total_orders invalid';
        }

        // Amount (3 decimals allowed)
        if ($total_amount === '' || !preg_match('/^\d+(\.\d{1,3})?$/', $total_amount)) {
          $errors[] = 'total_amount invalid';
        }

        // Uniqueness preview: existing row?
        if (!$errors && $valid_date && $cache_driver[$partner_id]) {
          $driver_id = (int)$cache_driver[$partner_id]['id'];
          $chk = $conn->prepare("SELECT id FROM partner_daily WHERE service_date=? AND driver_id=? LIMIT 1");
          $chk->bind_param('si', $date, $driver_id);
          $chk->execute();
          if ($chk->get_result()->fetch_assoc()) {
            $errors[] = 'duplicate row exists (service_date+driver_id)';
          }
        }

        $status = $errors ? 'error' : 'ok';
        if ($status==='ok') {
          $stats['ok']++;
          $ok_rows[] = [
            'service_date' => $date,
            'driver_id'    => (int)$cache_driver[$partner_id]['id'],
            'project_id'   => (int)$cache_driver[$partner_id]['project_id'],
            'partner_id'   => $partner_id,
            'driver_name'  => $cache_driver[$partner_id]['name'],
            'total_orders' => (int)$total_orders,
            'total_amount' => (float)$total_amount,
            'remarks'      => $remarks,
          ];
        } else {
          $stats['bad']++;
        }

        $preview_rows[] = [
          'row'         => $rownum,
          'date'        => $date,
          'partner_id'  => $partner_id,
          'driver_name' => $driver_name_csv,
          'orders'      => $total_orders,
          'amount'      => $total_amount,
          'remarks'     => $remarks,
          'status'      => $status,
          'errors'      => $errors,
          'resolved_name'=> $cache_driver[$partner_id]['name'] ?? ''
        ];
      }
      fclose($fh);
    }
  }
}

$payload_json = $ok_rows ? json_encode($ok_rows) : '';
$payload_sig  = $payload_json ? hmac_sign($payload_json, $csrf) : '';
?>
<!doctype html>
<html>
<head>
  <meta charset="utf-8">
  <title>Partner Daily Bulk Upload | Hisbuu</title>
  <meta name="viewport" content="width=device-width, initial-scale=1">
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
  <style>
    .status-ok{color:#0a7;font-weight:600}
    .status-err{color:#c00;font-weight:600}
    .mono{font-family:ui-monospace, SFMono-Regular, Menlo, Consolas, "Liberation Mono", monospace}
    .smallmuted{font-size:.9rem;color:#6c757d}
  </style>
</head>
<body class="bg-light">
<?php include __DIR__ . '/../../includes/admin_nav.php'; ?>

<div class="container py-4" style="max-width:1100px;">
  <div class="d-flex justify-content-between align-items-center mb-3">
    <h5 class="mb-0">Partner Daily, Bulk Upload</h5>
    <div class="d-flex gap-2">
      <a class="btn btn-outline-secondary btn-sm" href="/hisbuu/public/admin/data_entry.php">Back</a>
      <a class="btn btn-danger btn-sm" href="/hisbuu/public/admin/partner_upload.php?action=template">Download CSV template</a>
    </div>
  </div>

  <?php if($flash_ok): ?><div class="alert alert-success py-2"><?php echo e($flash_ok); ?></div><?php endif; ?>
  <?php if($flash_err): ?><div class="alert alert-danger py-2"><?php echo e($flash_err); ?></div><?php endif; ?>

  <div class="card mb-3">
    <div class="card-body">
      <form method="post" enctype="multipart/form-data" class="row g-2">
        <input type="hidden" name="csrf" value="<?php echo e($csrf); ?>">
        <input type="hidden" name="phase" value="preview">
        <div class="col-md-6">
          <label class="form-label">CSV file</label>
          <input type="file" name="csv" class="form-control" accept=".csv" required>
        </div>
        <div class="col-md-3 d-flex align-items-end">
          <div class="form-check">
            <input class="form-check-input" type="checkbox" name="has_header" id="has_header" checked>
            <label class="form-check-label" for="has_header">First row is header</label>
          </div>
        </div>
        <div class="col-md-3 d-flex align-items-end justify-content-end">
          <button class="btn btn-danger">Preview</button>
        </div>
      </form>
      <div class="smallmuted mt-2">
        Columns: date, driver_id (e.g., TB11070108), driver_name, total_orders, total_amount, remarks.  
        The system finds driver and project automatically. Amount supports 3 decimals.
      </div>
    </div>
  </div>

  <?php if ($is_preview && $preview_rows): ?>
    <div class="alert alert-info py-2">
      Checked <?php echo (int)$stats['total']; ?> rows. OK <?php echo (int)$stats['ok']; ?>. Errors <?php echo (int)$stats['bad']; ?>.
      <button class="btn btn-success btn-sm ms-2" data-bs-toggle="modal" data-bs-target="#confirmImportModal" <?php echo $stats['ok'] ? '' : 'disabled'; ?>>
        Review & Import OK rows
      </button>
    </div>

    <div class="card mb-4">
      <div class="card-body">
        <div class="table-responsive">
          <table class="table table-sm align-middle">
            <thead class="table-light">
              <tr>
                <th>#</th>
                <th>Date</th>
                <th>Driver ID</th>
                <th>CSV Name</th>
                <th>Resolved Name</th>
                <th>Orders</th>
                <th>Amount</th>
                <th>Status</th>
                <th>Errors</th>
              </tr>
            </thead>
            <tbody>
              <?php foreach ($preview_rows as $r): ?>
                <tr>
                  <td><?php echo (int)$r['row']; ?></td>
                  <td class="mono"><?php echo e($r['date']); ?></td>
                  <td class="mono"><?php echo e($r['partner_id']); ?></td>
                  <td><?php echo e($r['driver_name']); ?></td>
                  <td><?php echo e($r['resolved_name']); ?></td>
                  <td><?php echo e($r['orders']); ?></td>
                  <td><?php echo e($r['amount']); ?></td>
                  <td><?php echo $r['status']==='ok' ? '<span class="status-ok">OK</span>' : '<span class="status-err">ERROR</span>'; ?></td>
                  <td class="text-danger small"><?php echo $r['errors'] ? e(implode('; ',$r['errors'])) : ''; ?></td>
                </tr>
              <?php endforeach; ?>
            </tbody>
          </table>
          <div class="smallmuted">
            Nothing is saved yet. Fix your CSV and re-upload as needed.
          </div>
        </div>
      </div>
    </div>

    <!-- Confirm Import Modal -->
    <div class="modal fade" id="confirmImportModal" tabindex="-1" aria-hidden="true">
      <div class="modal-dialog modal-xl modal-dialog-centered">
        <div class="modal-content">
          <form method="post">
            <div class="modal-header">
              <h6 class="modal-title">Import confirmation</h6>
              <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
              <p class="mb-2">These rows will be imported:</p>
              <div class="table-responsive" style="max-height:50vh">
                <table class="table table-sm">
                  <thead class="table-light">
                    <tr>
                      <th>Date</th>
                      <th>Driver ID</th>
                      <th>Name</th>
                      <th>Orders</th>
                      <th>Amount</th>
                      <th>Remarks</th>
                    </tr>
                  </thead>
                  <tbody>
                    <?php foreach ($ok_rows as $r): ?>
                      <tr>
                        <td class="mono"><?php echo e($r['service_date']); ?></td>
                        <td class="mono"><?php echo e($r['partner_id']); ?></td>
                        <td><?php echo e($r['driver_name']); ?></td>
                        <td><?php echo (int)$r['total_orders']; ?></td>
                        <td><?php echo number_format((float)$r['total_amount'], 3, '.', ''); ?></td>
                        <td><?php echo e($r['remarks']); ?></td>
                      </tr>
                    <?php endforeach; ?>
                  </tbody>
                </table>
              </div>
              <div class="alert alert-warning small mt-3 mb-0">
                Only OK rows are inserted. Duplicates for the same date and driver are skipped.
              </div>
            </div>
            <div class="modal-footer">
              <input type="hidden" name="csrf" value="<?php echo e($csrf); ?>">
              <input type="hidden" name="phase" value="import">
              <input type="hidden" name="payload" value="<?php echo htmlspecialchars($payload_json, ENT_QUOTES, 'UTF-8'); ?>">
              <input type="hidden" name="sig" value="<?php echo htmlspecialchars($payload_sig, ENT_QUOTES, 'UTF-8'); ?>">
              <button type="button" class="btn btn-light" data-bs-dismiss="modal">Cancel</button>
              <button class="btn btn-success" <?php echo $stats['ok'] ? '' : 'disabled'; ?>>Import</button>
            </div>
          </form>
        </div>
      </div>
    </div>
  <?php endif; ?>

</div>

<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>
