<?php
// public/admin/payouts/payout_cycle_view.php
require_once __DIR__ . '/../../../includes/admin_bootstrap.php';

$cycle_id = (int)($_GET['id'] ?? 0);
$error    = null;

// Load cycle
$stmt = $conn->prepare("SELECT * FROM payout_cycles WHERE id = ?");
$stmt->bind_param('i', $cycle_id);
$stmt->execute();
$res   = $stmt->get_result();
$cycle = $res->fetch_assoc();
$stmt->close();

if (!$cycle) {
    die('Payout cycle not found.');
}

// Handle actions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $user_id = (int)$_SESSION['user']['id'];

    // Recalculate all
    if (isset($_POST['recalculate_all']) && $cycle['status'] === 'draft') {
        header('Location: payout_recalculate.php?cycle_id=' . $cycle_id);
        exit;
    }

    // Lock
    if (isset($_POST['lock_cycle']) && $cycle['status'] === 'draft') {
        $stmt = $conn->prepare("
            UPDATE payout_cycles
            SET status = 'locked', locked_at = NOW()
            WHERE id = ? AND status = 'draft'
        ");
        $stmt->bind_param('i', $cycle_id);
        $stmt->execute();
        $affected = $stmt->affected_rows;
        $stmt->close();

        if ($affected > 0) {
            $info = 'Cycle locked';
            $stmt = $conn->prepare("
                INSERT INTO payout_audit_log
                    (cycle_id, driver_id, action, details, done_by, created_at)
                VALUES (?, NULL, 'lock_cycle', ?, ?, NOW())
            ");
            $stmt->bind_param('isi', $cycle_id, $info, $user_id);
            $stmt->execute();
            $stmt->close();

            header('Location: payout_cycle_view.php?id=' . $cycle_id);
            exit;
        } else {
            $error = 'Could not lock this cycle.';
        }
    }

    // Mark paid
    if (isset($_POST['mark_paid']) && $cycle['status'] === 'locked') {
        $stmt = $conn->prepare("
            UPDATE payout_cycles
            SET status = 'paid', paid_at = NOW()
            WHERE id = ? AND status = 'locked'
        ");
        $stmt->bind_param('i', $cycle_id);
        $stmt->execute();
        $affected = $stmt->affected_rows;
        $stmt->close();

        if ($affected > 0) {
            $info = 'Cycle marked as paid';
            $stmt = $conn->prepare("
                INSERT INTO payout_audit_log
                    (cycle_id, driver_id, action, details, done_by, created_at)
                VALUES (?, NULL, 'mark_paid', ?, ?, NOW())
            ");
            $stmt->bind_param('isi', $cycle_id, $info, $user_id);
            $stmt->execute();
            $stmt->close();

            header('Location: payout_cycle_view.php?id=' . $cycle_id);
            exit;
        } else {
            $error = 'Could not mark this cycle as paid.';
        }
    }

    // Reload cycle state
    $stmt = $conn->prepare("SELECT * FROM payout_cycles WHERE id = ?");
    $stmt->bind_param('i', $cycle_id);
    $stmt->execute();
    $res   = $stmt->get_result();
    $cycle = $res->fetch_assoc();
    $stmt->close();
}

// Summary totals from payout_items
$tot = [
    'drivers'      => 0,
    'earnings'     => 0.0,
    'pos'          => 0.0,
    'neg'          => 0.0,
    'adv'          => 0.0,
    'short'        => 0.0,
    'net'          => 0.0,
];

$stmt = $conn->prepare("
    SELECT
      COUNT(*)                                       AS drivers_count,
      COALESCE(SUM(total_earnings), 0)               AS total_earnings,
      COALESCE(SUM(total_positive_adjustments), 0)   AS total_pos,
      COALESCE(SUM(total_negative_adjustments), 0)   AS total_neg,
      COALESCE(SUM(total_advances), 0)               AS total_adv,
      COALESCE(SUM(total_shortages), 0)              AS total_short,
      COALESCE(SUM(net_payout), 0)                   AS total_net
    FROM payout_items
    WHERE cycle_id = ?
");
$stmt->bind_param('i', $cycle_id);
$stmt->execute();
$res  = $stmt->get_result();
$row  = $res->fetch_assoc();
$stmt->close();

if ($row) {
    $tot['drivers']  = (int)$row['drivers_count'];
    $tot['earnings'] = (float)$row['total_earnings'];
    $tot['pos']      = (float)$row['total_pos'];
    $tot['neg']      = (float)$row['total_neg'];
    $tot['adv']      = (float)$row['total_adv'];
    $tot['short']    = (float)$row['total_short'];
    $tot['net']      = (float)$row['total_net'];
}

// Driver rows
$sql = "
    SELECT
      pi.*,
      d.first_name,
      d.last_name
    FROM payout_items pi
    JOIN drivers d ON d.id = pi.driver_id
    WHERE pi.cycle_id = ?
    ORDER BY d.first_name, d.last_name, d.id
";
$stmt = $conn->prepare($sql);
$stmt->bind_param('i', $cycle_id);
$stmt->execute();
$res   = $stmt->get_result();
$items = [];
while ($r = $res->fetch_assoc()) {
    $items[] = $r;
}
$stmt->close();
?>
<!doctype html>
<html lang="en">
<head>
  <meta charset="utf-8">
  <title>Payout Cycle | Hisbuu</title>
  <meta name="viewport" content="width=device-width, initial-scale=1">
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
</head>
<body class="bg-light">

<?php include __DIR__ . '/../../../includes/admin_nav.php'; ?>

<div class="container py-3">

  <div class="d-flex justify-content-between align-items-center mb-3">
    <h5 class="mb-0">
      Payout Cycle
      <?php echo e($cycle['cycle_month']); ?>
      <small class="text-muted">
        (<?php echo e($cycle['period_start']); ?> to <?php echo e($cycle['period_end']); ?>)
      </small>
    </h5>
    <div class="d-flex gap-2">
      <a href="payouts.php" class="btn btn-outline-secondary btn-sm">Back to cycles</a>
    </div>
  </div>

  <?php if ($error): ?>
    <div class="alert alert-danger py-2"><?php echo e($error); ?></div>
  <?php endif; ?>

  <div class="card mb-3">
    <div class="card-body">
      <form method="post" class="d-flex flex-wrap gap-2">
        <?php if ($cycle['status'] === 'draft'): ?>
          <button type="submit" name="recalculate_all" class="btn btn-outline-primary btn-sm">
            Recalculate all drivers
          </button>
          <button type="submit" name="lock_cycle" class="btn btn-warning btn-sm"
                  onclick="return confirm('Lock this cycle? You will not be able to recalc after locking.');">
            Lock cycle
          </button>
        <?php endif; ?>

        <?php if ($cycle['status'] === 'locked'): ?>
          <button type="submit" name="mark_paid" class="btn btn-success btn-sm"
                  onclick="return confirm('Mark this cycle as paid?');">
            Mark as paid
          </button>
        <?php endif; ?>

        <span class="ms-auto small text-muted">
          Status:
          <strong><?php echo e($cycle['status']); ?></strong>
        </span>
      </form>
    </div>
  </div>

  <!-- Totals summary -->
  <div class="row g-3 mb-3">
    <div class="col-md-3">
      <div class="card border-secondary">
        <div class="card-body py-2">
          <div class="d-flex justify-content-between">
            <span class="fw-semibold">Drivers in cycle</span>
            <span><?php echo number_format($tot['drivers']); ?></span>
          </div>
          <div class="small text-muted">Unique drivers with payout items</div>
        </div>
      </div>
    </div>

    <div class="col-md-3">
      <div class="card border-success">
        <div class="card-body py-2">
          <div class="d-flex justify-content-between">
            <span class="fw-semibold">Total earnings</span>
            <span class="text-success"><?php echo number_format($tot['earnings'], 3); ?></span>
          </div>
          <div class="small text-muted">Order based earnings</div>
        </div>
      </div>
    </div>

    <div class="col-md-3">
      <div class="card border-success">
        <div class="card-body py-2">
          <div class="d-flex justify-content-between">
            <span class="fw-semibold">Total positives</span>
            <span class="text-success"><?php echo number_format($tot['pos'], 3); ?></span>
          </div>
          <div class="small text-muted">Bonuses and incentives</div>
        </div>
      </div>
    </div>

    <div class="col-md-3">
      <div class="card border-danger">
        <div class="card-body py-2">
          <div class="d-flex justify-content-between">
            <span class="fw-semibold">Total negatives</span>
            <span class="text-danger"><?php echo number_format($tot['neg'], 3); ?></span>
          </div>
          <div class="small text-muted">Deductions excluding advances and shortages</div>
        </div>
      </div>
    </div>

    <div class="col-md-3">
      <div class="card border-danger">
        <div class="card-body py-2">
          <div class="d-flex justify-content-between">
            <span class="fw-semibold">Total advances</span>
            <span class="text-danger"><?php echo number_format($tot['adv'], 3); ?></span>
          </div>
          <div class="small text-muted">Salary advances within this cycle</div>
        </div>
      </div>
    </div>

    <div class="col-md-3">
      <div class="card border-danger">
        <div class="card-body py-2">
          <div class="d-flex justify-content-between">
            <span class="fw-semibold">Total shortages</span>
            <span class="text-danger"><?php echo number_format($tot['short'], 3); ?></span>
          </div>
          <div class="small text-muted">COD and other shortages</div>
        </div>
      </div>
    </div>

    <div class="col-md-3">
      <div class="card border-primary">
        <div class="card-body py-2">
          <div class="d-flex justify-content-between">
            <span class="fw-semibold">Net payable</span>
            <span class="text-primary"><?php echo number_format($tot['net'], 3); ?></span>
          </div>
          <div class="small text-muted">Sum of driver net payouts</div>
        </div>
      </div>
    </div>
  </div>

  <!-- Driver table -->
  <div class="card">
    <div class="card-body">
      <div class="table-responsive">
        <table class="table table-striped align-middle">
          <thead class="table-danger">
          <tr>
            <th>Driver</th>
            <th class="text-end">Earnings</th>
            <th class="text-end">Pos adj</th>
            <th class="text-end">Neg adj</th>
            <th class="text-end">Advances</th>
            <th class="text-end">Shortages</th>
            <th class="text-end">Net payout</th>
            <th></th>
          </tr>
          </thead>
          <tbody>
          <?php if (!$items): ?>
            <tr>
              <td colspan="8" class="text-center text-muted">
                No payout items yet. Use Recalculate to build this cycle.
              </td>
            </tr>
          <?php else: ?>
            <?php foreach ($items as $row): ?>
              <?php
              $driver_name = trim(($row['first_name'] ?? '') . ' ' . ($row['last_name'] ?? ''));
              if ($driver_name === '') {
                  $driver_name = 'Driver #' . (int)$row['driver_id'];
              }
              ?>
              <tr>
                <td><?php echo e($driver_name); ?></td>
                <td class="text-end"><?php echo number_format((float)$row['total_earnings'], 3); ?></td>
                <td class="text-end text-success"><?php echo number_format((float)$row['total_positive_adjustments'], 3); ?></td>
                <td class="text-end text-danger"><?php echo number_format((float)$row['total_negative_adjustments'], 3); ?></td>
                <td class="text-end text-danger"><?php echo number_format((float)$row['total_advances'], 3); ?></td>
                <td class="text-end text-danger"><?php echo number_format((float)$row['total_shortages'], 3); ?></td>
                <td class="text-end fw-semibold"><?php echo number_format((float)$row['net_payout'], 3); ?></td>
                <td>
                  <a href="payout_driver_view.php?cycle_id=<?php echo $cycle_id; ?>&driver_id=<?php echo (int)$row['driver_id']; ?>"
                     class="btn btn-sm btn-outline-primary">
                    View
                  </a>
                  <?php if ($cycle['status'] === 'draft'): ?>
                    <a href="payout_recalculate.php?cycle_id=<?php echo $cycle_id; ?>&driver_id=<?php echo (int)$row['driver_id']; ?>"
                       class="btn btn-sm btn-outline-secondary">
                      Recalc
                    </a>
                  <?php endif; ?>
                </td>
              </tr>
            <?php endforeach; ?>
          <?php endif; ?>
          </tbody>
        </table>
      </div>
    </div>
  </div>

</div>

</body>
</html>
