<?php
// public/admin/payouts/payout_recalculate.php
require_once __DIR__ . '/../../../includes/admin_bootstrap.php';

$cycle_id  = (int)($_GET['cycle_id'] ?? 0);
$driver_id = (int)($_GET['driver_id'] ?? 0);

// Load cycle
$stmt = $conn->prepare("SELECT * FROM payout_cycles WHERE id = ?");
$stmt->bind_param('i', $cycle_id);
$stmt->execute();
$res   = $stmt->get_result();
$cycle = $res->fetch_assoc();
$stmt->close();

if (!$cycle) {
    die('Payout cycle not found.');
}
if ($cycle['status'] !== 'draft') {
    die('Only draft cycles can be recalculated.');
}

$period_start = $cycle['period_start'];
$period_end   = $cycle['period_end'];
$cycle_month  = $cycle['cycle_month'];

$current_user_id = (int)$_SESSION['user']['id'];

// Decide which drivers to process
$driverIds = [];

if ($driver_id > 0) {
    $driverIds = [$driver_id];
} else {
    $ids = [];

    // Drivers that have monthly earnings in this month
    $stmt = $conn->prepare("
        SELECT DISTINCT driver_id
        FROM driver_earnings_monthly
        WHERE month = ?
    ");
    $stmt->bind_param('s', $cycle_month);
    $stmt->execute();
    $res = $stmt->get_result();
    while ($row = $res->fetch_assoc()) {
        $ids[] = (int)$row['driver_id'];
    }
    $stmt->close();

    // Drivers that have adjustments affecting this cycle
    $stmt = $conn->prepare("
        SELECT DISTINCT driver_id
        FROM driver_adjustments
        WHERE (applied_in_month = ?)
           OR (applied_in_month IS NULL AND adj_date BETWEEN ? AND ?)
    ");
    $stmt->bind_param('sss', $cycle_month, $period_start, $period_end);
    $stmt->execute();
    $res = $stmt->get_result();
    while ($row = $res->fetch_assoc()) {
        $ids[] = (int)$row['driver_id'];
    }
    $stmt->close();

    $driverIds = array_values(array_unique($ids));
}

if (!$driverIds) {
    header('Location: payout_cycle_view.php?id=' . $cycle_id);
    exit;
}

foreach ($driverIds as $dId) {
    $dId = (int)$dId;
    if ($dId <= 0) {
        continue;
    }

    // Earnings from driver_earnings_monthly for this month
    $stmt = $conn->prepare("
        SELECT
          COALESCE(SUM(completed_deliveries), 0) AS completed_deliveries,
          COALESCE(SUM(operations_fee), 0)       AS operations_fee,
          COALESCE(SUM(tips), 0)                 AS tips,
          COALESCE(SUM(incentives), 0)           AS incentives,
          COALESCE(SUM(compensations), 0)        AS compensations,
          COALESCE(SUM(deductions), 0)           AS deductions,
          COALESCE(SUM(total_earning), 0)        AS total_earning
        FROM driver_earnings_monthly
        WHERE driver_id = ? AND month = ?
    ");
    $stmt->bind_param('is', $dId, $cycle_month);
    $stmt->execute();
    $res     = $stmt->get_result();
    $earnRow = $res->fetch_assoc();
    $stmt->close();

    $completed_deliveries = (int)($earnRow['completed_deliveries'] ?? 0);
    $operations_fee       = (float)($earnRow['operations_fee'] ?? 0);
    $tips                 = (float)($earnRow['tips'] ?? 0);
    $incentives           = (float)($earnRow['incentives'] ?? 0);
    $compensations        = (float)($earnRow['compensations'] ?? 0);
    $deductions           = (float)($earnRow['deductions'] ?? 0);
    $earnings             = (float)($earnRow['total_earning'] ?? 0);

    // Adjustments affecting this cycle
    $stmt = $conn->prepare("
        SELECT *
        FROM driver_adjustments
        WHERE driver_id = ?
          AND (
                applied_in_month = ?
             OR (applied_in_month IS NULL AND adj_date BETWEEN ? AND ?)
          )
    ");
    $stmt->bind_param('isss', $dId, $cycle_month, $period_start, $period_end);
    $stmt->execute();
    $res         = $stmt->get_result();
    $adjustments = [];
    while ($row = $res->fetch_assoc()) {
        $adjustments[] = $row;
    }
    $stmt->close();

    $total_pos   = 0.0;
    $total_neg   = 0.0;
    $total_adv   = 0.0;
    $total_short = 0.0;
    $breakdown   = [];

    // Earnings breakdown row
    if ($earnings != 0.0) {
        $labelParts = [];
        $labelParts[] = 'Ops ' . number_format($operations_fee, 3);
        if ($tips != 0.0) {
            $labelParts[] = 'Tips ' . number_format($tips, 3);
        }
        if ($incentives != 0.0) {
            $labelParts[] = 'Incentives ' . number_format($incentives, 3);
        }
        if ($compensations != 0.0) {
            $labelParts[] = 'Comp ' . number_format($compensations, 3);
        }
        if ($deductions != 0.0) {
            $labelParts[] = 'Deductions ' . number_format($deductions, 3);
        }

        $label = 'Monthly earnings';
        if ($completed_deliveries > 0) {
            $label .= ' (' . $completed_deliveries . ' completed deliveries)';
        }
        if ($labelParts) {
            $label .= ': ' . implode(', ', $labelParts);
        }

        $breakdown[] = [
            'type'           => 'earning',
            'reference_type' => 'monthly_aggregator',
            'reference_id'   => null,
            'label'          => $label,
            'amount'         => $earnings,
        ];
    }

    // Adjustments breakdown rows
    foreach ($adjustments as $adj) {
        $amount_raw = (float)$adj['amount'];
        $label      = $adj['remarks'] ?: ('Adjustment #' . (int)$adj['id']);
        $type       = 'adjustment';
        $signed     = 0.0;

        $is_advance  = (int)$adj['is_advance'] === 1;
        $is_shortage = (int)$adj['is_shortage'] === 1;
        $adj_type    = $adj['type']; // positive or negative

        if ($is_advance) {
            $type       = 'advance';
            $signed     = -$amount_raw;
            $total_adv += $amount_raw;
        } elseif ($is_shortage) {
            $type        = 'shortage';
            $signed      = -$amount_raw;
            $total_short += $amount_raw;
        } else {
            if ($adj_type === 'positive') {
                $type      = 'adjustment';
                $signed    = $amount_raw;
                $total_pos += $amount_raw;
            } else {
                $type      = 'adjustment';
                $signed    = -$amount_raw;
                $total_neg += $amount_raw;
            }
        }

        $breakdown[] = [
            'type'           => $type,
            'reference_type' => 'adjustment',
            'reference_id'   => (int)$adj['id'],
            'label'          => $label,
            'amount'         => $signed,
        ];
    }

    // Net payout
    $net_payout = $earnings + $total_pos - $total_neg - $total_adv - $total_short;

    // Snapshot JSON
    $snapshot = json_encode([
        'driver_id'            => $dId,
        'cycle_id'             => $cycle_id,
        'cycle_month'          => $cycle_month,
        'period_start'         => $period_start,
        'period_end'           => $period_end,
        'completed_deliveries' => $completed_deliveries,
        'operations_fee'       => $operations_fee,
        'tips'                 => $tips,
        'incentives'           => $incentives,
        'compensations'        => $compensations,
        'deductions'           => $deductions,
        'earnings'             => $earnings,
        'pos'                  => $total_pos,
        'neg'                  => $total_neg,
        'adv'                  => $total_adv,
        'short'                => $total_short,
        'net'                  => $net_payout,
    ]);

    $conn->begin_transaction();

    try {
        // Check existing payout_item
        $stmt = $conn->prepare("
            SELECT id
            FROM payout_items
            WHERE cycle_id = ? AND driver_id = ?
            LIMIT 1
        ");
        $stmt->bind_param('ii', $cycle_id, $dId);
        $stmt->execute();
        $res      = $stmt->get_result();
        $existing = $res->fetch_assoc();
        $stmt->close();

        $payout_item_id = null;

        if ($existing) {
            $payout_item_id = (int)$existing['id'];

            $stmt = $conn->prepare("
                UPDATE payout_items
                SET
                  total_earnings             = ?,
                  total_positive_adjustments = ?,
                  total_negative_adjustments = ?,
                  total_advances             = ?,
                  total_shortages            = ?,
                  net_payout                 = ?,
                  calculation_snapshot       = ?,
                  updated_at                 = NOW()
                WHERE id = ?
            ");
            $stmt->bind_param(
                'ddddddsi',
                $earnings,
                $total_pos,
                $total_neg,
                $total_adv,
                $total_short,
                $net_payout,
                $snapshot,
                $payout_item_id
            );
            $stmt->execute();
            $stmt->close();

            // Clear old breakdowns
            $stmt = $conn->prepare("DELETE FROM payout_breakdowns WHERE payout_item_id = ?");
            $stmt->bind_param('i', $payout_item_id);
            $stmt->execute();
            $stmt->close();
        } else {
            $stmt = $conn->prepare("
                INSERT INTO payout_items
                    (cycle_id, driver_id,
                     total_earnings, total_positive_adjustments, total_negative_adjustments,
                     total_advances, total_shortages, net_payout,
                     calculation_snapshot, created_at)
                VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, NOW())
            ");
            // types: i, i, d, d, d, d, d, d, s
            $typeStr = 'iidddddds';
            $stmt->bind_param(
                $typeStr,
                $cycle_id,
                $dId,
                $earnings,
                $total_pos,
                $total_neg,
                $total_adv,
                $total_short,
                $net_payout,
                $snapshot
            );
            $stmt->execute();
            $payout_item_id = $stmt->insert_id;
            $stmt->close();
        }

        // Insert breakdown rows
        if ($payout_item_id) {
            $stmt = $conn->prepare("
                INSERT INTO payout_breakdowns
                    (payout_item_id, type, reference_type, reference_id, label, amount, created_at)
                VALUES (?, ?, ?, ?, ?, ?, NOW())
            ");

            foreach ($breakdown as $b) {
                $ref_id     = $b['reference_id'];
                $ref_type   = $b['reference_type'];
                $type       = $b['type'];
                $label      = $b['label'];
                $amount_val = (float)$b['amount'];

                if ($ref_id === null) {
                    $ref_id = null;
                } else {
                    $ref_id = (int)$ref_id;
                }

                $stmt->bind_param(
                    'issisd',
                    $payout_item_id,
                    $type,
                    $ref_type,
                    $ref_id,
                    $label,
                    $amount_val
                );
                $stmt->execute();
            }
            $stmt->close();
        }

        // Audit log
        $details = 'Recalculated driver ' . $dId;
        $stmt = $conn->prepare("
            INSERT INTO payout_audit_log
                (cycle_id, driver_id, action, details, done_by, created_at)
            VALUES (?, ?, 'recalculate', ?, ?, NOW())
        ");
        $stmt->bind_param('iisi', $cycle_id, $dId, $details, $current_user_id);
        $stmt->execute();
        $stmt->close();

        $conn->commit();
    } catch (Exception $e) {
        $conn->rollback();
        // Optional: log $e->getMessage()
        continue;
    }
}

header('Location: payout_cycle_view.php?id=' . $cycle_id);
exit;
