<?php
// public/admin/pending_approvals.php
require_once __DIR__ . '/../../includes/admin_bootstrap.php';
$role = strtolower(get_current_role() ?? '');

$can_view = in_array($role, ['super administrator','administrator','finance administrator','supervisor'], true);
if (!$can_view) { http_response_code(403); exit('Forbidden'); }
$can_act  = in_array($role, ['super administrator','administrator','finance administrator'], true);

if (session_status() === PHP_SESSION_NONE) session_start();
if (empty($_SESSION['csrf'])) $_SESSION['csrf'] = bin2hex(random_bytes(16));
$csrf = $_SESSION['csrf'];

$flash_ok=''; $flash_err='';

function badge_html($status) {
  $map = [
    'pending'  => 'warning',
    'approved' => 'success',
    'rejected' => 'danger',
  ];
  $cls = $map[strtolower((string)$status)] ?? 'secondary';
  $label = htmlspecialchars(ucfirst($status), ENT_QUOTES, 'UTF-8');
  return '<span class="badge text-bg-'.$cls.'">'.$label.'</span>';
}

function render_media_preview(?string $path, bool $fullSize = false, bool $inGrid = false): string {
  if (!$path) return '<div class="text-muted text-center py-5"><i class="bi bi-image" style="font-size: 48px; opacity: 0.3;"></i><div class="mt-2 small">No file uploaded</div></div>';
  $url = file_url($path);

  // Get extension from the actual path, not parsed URL
  $ext = strtolower(pathinfo($path, PATHINFO_EXTENSION));

  // Check if it's an image - be more lenient with extension checking
  $imageExts = ['jpg','jpeg','png','gif','webp','bmp','svg'];
  $isImage = in_array($ext, $imageExts);

  // If extension check fails, assume it's an image (mobile screenshots)
  if (!$ext || strlen($ext) > 5) {
    $isImage = true;
  }

  if ($isImage) {
    if ($inGrid) {
      return '<img src="'.e($url).'" alt="preview">';
    }
    if ($fullSize) {
      return '<img src="'.e($url).'" alt="preview">';
    }
    return '<img src="'.e($url).'" class="img-fluid rounded" alt="preview" style="max-height: 400px; object-fit: contain;">';
  }

  if ($ext === 'pdf') {
    return '<div class="ratio ratio-4x3"><iframe src="'.e($url).'" class="rounded border" allowfullscreen></iframe></div>';
  }

  return '<a class="btn btn-outline-secondary" href="'.e($url).'" target="_blank"><i class="bi bi-download"></i> Download file</a>';
}

function render_thumbnail(?string $path): string {
  if (!$path) return '<div class="bg-light border rounded d-flex align-items-center justify-content-center" style="width:60px;height:60px;"><i class="bi bi-image text-muted"></i></div>';
  $url = file_url($path);
  return '<img src="'.e($url).'" class="border rounded" style="width:60px;height:60px;object-fit:cover;" alt="thumb">';
}

// --- actions ---
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['csrf']) && hash_equals($csrf, $_POST['csrf'])) {
  $sid = isset($_POST['submission_id']) ? (int)$_POST['submission_id'] : 0;
  $action = $_POST['action'] ?? '';
  if ($sid <= 0) {
    $flash_err = 'Invalid submission ID.';
  } elseif (!$can_act && in_array($action, ['approve','reject','update'], true)) {
    $flash_err = 'You do not have permission for this action.';
  } elseif ($action === 'reject') {
    $reason = trim($_POST['reason'] ?? '');
    if ($reason === '') $reason = 'Rejected';
    $uid = (int)current_user_id();
    $stmt = $conn->prepare("UPDATE submissions SET status='rejected', reviewed_by=?, reviewed_at=NOW(), remarks=? WHERE id=? AND status<>'approved'");
    $stmt->bind_param('isi', $uid, $reason, $sid);
    if ($stmt->execute() && $stmt->affected_rows) $flash_ok = 'Submission rejected.';
    else $flash_err = 'Reject failed.';
  } elseif (in_array($action, ['update','approve'], true)) {
    $orders = trim($_POST['orders_count'] ?? '');
    $amount = trim($_POST['cash_deposited'] ?? '');
    $trx    = trim($_POST['trx_no'] ?? '');
    $remarks= trim($_POST['remarks'] ?? '');

    if ($orders === '' || !ctype_digit($orders)) $flash_err = 'Orders must be a whole number.';
    elseif ($amount === '' || !preg_match('/^\d+(\.\d{1,3})?$/', $amount)) $flash_err = 'Amount must be numeric with up to 3 decimals.';

    if (!$flash_err) {
      $stmt = $conn->prepare("UPDATE submissions SET orders_count=?, cash_deposited=?, trx_no=?, remarks=?, updated_at=NOW() WHERE id=? AND status<>'approved'");
      $stmt->bind_param('idssi', $orders, $amount, $trx, $remarks, $sid);
      if (!$stmt->execute()) {
        $flash_err = 'Update failed.';
      } else {
        if ($action === 'approve') {
          $uid = (int)current_user_id();
          $ap = $conn->prepare("UPDATE submissions SET status='approved', reviewed_by=?, reviewed_at=NOW() WHERE id=? AND status<>'approved'");
          $ap->bind_param('ii', $uid, $sid);
          if ($ap->execute() && $ap->affected_rows) $flash_ok = 'Submission approved.';
          else $flash_err = 'Approve failed.';
        } else {
          $flash_ok = 'Submission updated.';
        }
      }
    }
  }
}

// --- filters ---
$status = $_GET['status'] ?? 'pending';
$driver = trim($_GET['driver'] ?? '');
$from   = trim($_GET['from'] ?? '');
$to     = trim($_GET['to'] ?? '');

$where = []; $params = []; $types = '';
if ($status !== 'all') { $where[] = 's.status=?'; $types .= 's'; $params[] = $status; }
if ($driver !== '') {
  $where[] = "(d.partner_id LIKE CONCAT('%',?,'%') OR d.partner_uuid LIKE CONCAT('%',?,'%') OR CONCAT(d.first_name,' ',d.last_name) LIKE CONCAT('%',?,'%'))";
  $types .= 'sss'; $params[] = $driver; $params[] = $driver; $params[] = $driver;
}
if (preg_match('/^\d{4}-\d{2}-\d{2}$/', $from)) { $where[]='s.service_date>=?'; $types.='s'; $params[]=$from; }
if (preg_match('/^\d{4}-\d{2}-\d{2}$/', $to))   { $where[]='s.service_date<=?'; $types.='s'; $params[]=$to; }
$wsql = $where ? 'WHERE '.implode(' AND ', $where) : '';

$sql = "
SELECT
  s.id, s.driver_id, s.service_date, s.orders_count, s.cash_deposited,
  s.trx_no, s.bank_trx_no, s.bank_validated, s.bank_amount, s.bank_date,
  s.ss_orders, s.ss_deposit_1, s.ss_deposit_2, s.remarks, s.status, s.created_at,
  d.partner_id, d.partner_uuid, CONCAT(d.first_name,' ',d.last_name) AS driver_name,
  pd.partner_id AS partner_data_id, pd.partner_name, pd.orders_count AS partner_orders, pd.amount_total AS partner_amount
FROM submissions s
JOIN drivers d ON d.id = s.driver_id
LEFT JOIN partner_daily pd ON pd.driver_id = s.driver_id AND pd.service_date = s.service_date
$wsql
ORDER BY s.service_date DESC, s.created_at DESC
LIMIT 200";
$stmt = $conn->prepare($sql);
if ($types !== '') $stmt->bind_param($types, ...$params);
$stmt->execute();
$rows = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);

?>
<!doctype html>
<html lang="en">
<head>
  <meta charset="utf-8">
  <title>Pending approvals | Hisbuu</title>
  <meta name="viewport" content="width=device-width, initial-scale=1">
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
  <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.3/font/bootstrap-icons.min.css" rel="stylesheet">
  <style>
    .screenshot-preview {
      cursor: pointer;
      transition: transform 0.2s, box-shadow 0.2s;
    }
    .screenshot-preview:hover {
      transform: scale(1.05);
      box-shadow: 0 4px 12px rgba(0,0,0,0.15);
    }
    .modal-image-viewer {
      background: #000;
      display: flex;
      align-items: center;
      justify-content: center;
      max-height: 80vh;
      padding: 20px;
    }
    .modal-image-viewer img {
      max-height: 80vh;
      max-width: 100%;
      width: auto;
      height: auto;
      object-fit: contain;
    }
    .badge-file {
      display: inline-flex;
      align-items: center;
      gap: 4px;
    }
    .trx-badge {
      font-family: 'Courier New', monospace;
      font-weight: 600;
    }
    .screenshot-grid {
      display: grid;
      grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));
      gap: 16px;
    }
    .screenshot-card {
      border: 2px solid #e2e8f0;
      border-radius: 12px;
      overflow: hidden;
      background: #fff;
      transition: border-color 0.2s;
    }
    .screenshot-card:hover {
      border-color: #dc3545;
    }
    .screenshot-card-header {
      background: #f8fafc;
      padding: 10px 14px;
      font-weight: 600;
      font-size: 13px;
      color: #475569;
      border-bottom: 1px solid #e2e8f0;
    }
    .screenshot-card-body {
      padding: 0;
      height: 350px;
      display: flex;
      align-items: center;
      justify-content: center;
      background: #f8fafc;
      overflow: hidden;
      position: relative;
    }
    .screenshot-card-body img {
      width: 100%;
      height: 100%;
      object-fit: contain;
      padding: 12px;
    }
    .info-badge {
      background: #f1f5f9;
      border: 1px solid #e2e8f0;
      border-radius: 8px;
      padding: 12px 16px;
      display: flex;
      align-items: center;
      gap: 12px;
    }
    .info-badge i {
      font-size: 24px;
      color: #64748b;
    }
  </style>
</head>
<body class="bg-light">
<?php include __DIR__ . '/../../includes/admin_nav.php'; ?>

<div class="container py-4">
  <div class="d-flex justify-content-between align-items-center mb-3">
    <h5 class="mb-0">Driver submissions</h5>
    <div class="text-muted small">Showing latest 200 records</div>
  </div>

  <?php if ($flash_ok): ?><div class="alert alert-success"><?php echo e($flash_ok); ?></div><?php endif; ?>
  <?php if ($flash_err): ?><div class="alert alert-danger"><?php echo e($flash_err); ?></div><?php endif; ?>

  <div class="card mb-3">
    <div class="card-body">
      <form class="row g-2 align-items-end">
        <div class="col-auto">
          <label class="form-label">Status</label>
          <select class="form-select form-select-sm" name="status" onchange="this.form.submit()">
            <?php foreach (['all','pending','approved','rejected'] as $opt): ?>
              <option value="<?php echo $opt; ?>" <?php echo $status===$opt?'selected':''; ?>>
                <?php echo ucfirst($opt); ?>
              </option>
            <?php endforeach; ?>
          </select>
        </div>
        <div class="col-auto">
          <label class="form-label">Driver</label>
          <input class="form-control form-control-sm" name="driver" value="<?php echo e($driver); ?>" placeholder="name, ID or UUID">
        </div>
        <div class="col-auto">
          <label class="form-label">From</label>
          <input type="date" class="form-control form-control-sm" name="from" value="<?php echo e($from); ?>">
        </div>
        <div class="col-auto">
          <label class="form-label">To</label>
          <input type="date" class="form-control form-control-sm" name="to" value="<?php echo e($to); ?>">
        </div>
        <div class="col-auto">
          <button class="btn btn-primary btn-sm">Filter</button>
        </div>
      </form>
    </div>
  </div>

  <div class="card">
    <div class="card-body">
      <div class="table-responsive">
        <table class="table table-striped table-sm align-middle mb-0">
          <thead>
            <tr>
              <th>Driver</th>
              <th>Partner UUID</th>
              <th>Date</th>
              <th>Orders</th>
              <th>Amount</th>
              <th>TRX</th>
              <th>Files</th>
              <th>Status</th>
              <th class="text-end">Action</th>
            </tr>
          </thead>
          <tbody>
            <?php if (!$rows): ?>
              <tr><td colspan="9" class="text-center text-muted py-4">No submissions found</td></tr>
            <?php else: foreach ($rows as $r): $sid=(int)$r['id']; ?>
              <tr>
                <td>
                  <strong><?php echo e($r['driver_name']); ?></strong><br>
                  <small class="text-muted"><?php echo e($r['partner_id']); ?></small>
                </td>
                <td class="small"><?php echo e($r['partner_uuid']); ?></td>
                <td class="nowrap"><?php echo e($r['service_date']); ?></td>
                <td><?php echo (int)$r['orders_count']; ?></td>
                <td><?php echo number_format((float)$r['cash_deposited'], 3, '.', ''); ?></td>
                <td>
                  <?php
                    $bankTrx   = trim((string)($r['bank_trx_no'] ?? ''));
                    $driverTrx = trim((string)($r['trx_no'] ?? ''));
                  ?>
                  <?php if ($bankTrx !== ''): ?>
                    <div><strong><?php echo e($bankTrx); ?></strong></div>
                    <div class="text-success small">TRX matched</div>
                    <?php if (!is_null($r['bank_amount']) || !is_null($r['bank_date'])): ?>
                      <div class="text-muted small">
                        <?php
                          $bits = [];
                          if (!is_null($r['bank_amount'])) $bits[] = 'OMR '.number_format((float)$r['bank_amount'],3);
                          if (!is_null($r['bank_date']))   $bits[] = e($r['bank_date']);
                          echo implode(' · ', $bits);
                        ?>
                      </div>
                    <?php endif; ?>
                  <?php elseif ($driverTrx !== ''): ?>
                    <div><strong><?php echo e($driverTrx); ?></strong></div>
                    <div class="text-muted small">Driver provided</div>
                  <?php else: ?>
                    <span class="badge text-bg-light">No TRX</span>
                  <?php endif; ?>
                </td>
                <td class="nowrap">
                  <div class="btn-group btn-group-sm" role="group">
                    <button class="btn btn-outline-primary" data-bs-toggle="modal" data-bs-target="#mOrders_<?php echo $sid; ?>" title="View order screenshot">
                      <i class="bi bi-card-list"></i> Order
                    </button>
                    <button class="btn btn-outline-info" data-bs-toggle="modal" data-bs-target="#mDep1_<?php echo $sid; ?>" title="View deposit slip 1">
                      <i class="bi bi-receipt"></i> Dep 1
                    </button>
                    <button class="btn btn-outline-info" data-bs-toggle="modal" data-bs-target="#mDep2_<?php echo $sid; ?>" title="View deposit slip 2">
                      <i class="bi bi-receipt"></i> Dep 2
                    </button>
                  </div>
                </td>
                <td><?php echo badge_html($r['status']); ?></td>
                <td class="text-end">
                  <?php if ($can_act && strtolower($r['status'])!=='approved'): ?>
                    <button class="btn btn-success btn-sm" data-bs-toggle="modal" data-bs-target="#mApprove_<?php echo $sid; ?>">Approve</button>
                    <button class="btn btn-outline-danger btn-sm" data-bs-toggle="modal" data-bs-target="#mReject_<?php echo $sid; ?>">Reject</button>
                  <?php else: ?>
                    <span class="text-muted">No actions</span>
                  <?php endif; ?>
                </td>
              </tr>

              <!-- Image Preview Modals -->
              <div class="modal fade" id="mOrders_<?php echo $sid; ?>" tabindex="-1" aria-hidden="true">
                <div class="modal-dialog modal-xl modal-dialog-centered">
                  <div class="modal-content">
                    <div class="modal-header bg-dark text-white">
                      <h6 class="modal-title"><i class="bi bi-card-list"></i> Order Screenshot · <?php echo e($r['service_date']); ?></h6>
                      <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal"></button>
                    </div>
                    <div class="modal-body modal-image-viewer p-0">
                      <?php echo render_media_preview($r['ss_orders'] ?? null, true); ?>
                    </div>
                  </div>
                </div>
              </div>
              <div class="modal fade" id="mDep1_<?php echo $sid; ?>" tabindex="-1" aria-hidden="true">
                <div class="modal-dialog modal-xl modal-dialog-centered">
                  <div class="modal-content">
                    <div class="modal-header bg-dark text-white">
                      <h6 class="modal-title"><i class="bi bi-receipt"></i> Deposit Slip 1 · <?php echo e($r['service_date']); ?></h6>
                      <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal"></button>
                    </div>
                    <div class="modal-body modal-image-viewer p-0">
                      <?php echo render_media_preview($r['ss_deposit_1'] ?? null, true); ?>
                    </div>
                  </div>
                </div>
              </div>
              <div class="modal fade" id="mDep2_<?php echo $sid; ?>" tabindex="-1" aria-hidden="true">
                <div class="modal-dialog modal-xl modal-dialog-centered">
                  <div class="modal-content">
                    <div class="modal-header bg-dark text-white">
                      <h6 class="modal-title"><i class="bi bi-receipt"></i> Deposit Slip 2 · <?php echo e($r['service_date']); ?></h6>
                      <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal"></button>
                    </div>
                    <div class="modal-body modal-image-viewer p-0">
                      <?php echo render_media_preview($r['ss_deposit_2'] ?? null, true); ?>
                    </div>
                  </div>
                </div>
              </div>

              <div class="modal fade" id="mApprove_<?php echo $sid; ?>" tabindex="-1" aria-hidden="true">
                <div class="modal-dialog modal-fullscreen">
                  <div class="modal-content">
                    <div class="modal-header bg-light">
                      <div>
                        <h5 class="modal-title mb-1">
                          <i class="bi bi-check-circle text-success"></i> Review & Approve Submission
                        </h5>
                        <div class="d-flex gap-3 flex-wrap small text-muted">
                          <span><i class="bi bi-person"></i> <strong><?php echo e($r['driver_name']); ?></strong></span>
                          <span><i class="bi bi-calendar3"></i> <?php echo e($r['service_date']); ?></span>
                          <span><i class="bi bi-truck"></i> <?php echo e($r['partner_id']); ?></span>
                        </div>
                      </div>
                      <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                    </div>

                    <div class="modal-body bg-light" style="max-width: 1400px; margin: 0 auto;">
                      <form method="post" id="formApprove_<?php echo $sid; ?>">
                        <input type="hidden" name="csrf" value="<?php echo e($csrf); ?>">
                        <input type="hidden" name="submission_id" value="<?php echo $sid; ?>">

                        <!-- Partner Data (Read-only Reference) -->
                        <?php if (!empty($r['partner_data_id']) || !empty($r['partner_name'])): ?>
                          <div class="card mb-4 shadow-sm border-primary">
                            <div class="card-header bg-primary bg-opacity-10 border-primary">
                              <h6 class="mb-0 text-primary">
                                <i class="bi bi-building-check me-2"></i>Partner Data (Reference Only)
                              </h6>
                            </div>
                            <div class="card-body">
                              <div class="alert alert-info mb-3">
                                <i class="bi bi-info-circle me-2"></i>
                                This data is imported from the partner system and provided for reference only. You cannot edit these values here.
                              </div>
                              <div class="row g-3">
                                <?php if (!empty($r['partner_name'])): ?>
                                <div class="col-md-4">
                                  <div class="border rounded p-3 bg-white">
                                    <div class="text-muted small mb-1">
                                      <i class="bi bi-building me-1"></i>Partner Name
                                    </div>
                                    <div class="fw-bold"><?php echo e($r['partner_name']); ?></div>
                                  </div>
                                </div>
                                <?php endif; ?>

                                <?php if (!empty($r['partner_data_id'])): ?>
                                <div class="col-md-4">
                                  <div class="border rounded p-3 bg-white">
                                    <div class="text-muted small mb-1">
                                      <i class="bi bi-hash me-1"></i>Partner ID
                                    </div>
                                    <div class="fw-bold font-monospace"><?php echo e($r['partner_data_id']); ?></div>
                                  </div>
                                </div>
                                <?php endif; ?>

                                <?php if (isset($r['partner_orders']) && $r['partner_orders'] !== null): ?>
                                <div class="col-md-2">
                                  <div class="border rounded p-3 bg-white">
                                    <div class="text-muted small mb-1">
                                      <i class="bi bi-bag-check me-1"></i>Orders Count
                                    </div>
                                    <div class="fw-bold fs-5 text-success"><?php echo (int)$r['partner_orders']; ?></div>
                                  </div>
                                </div>
                                <?php endif; ?>

                                <?php if (isset($r['partner_amount']) && $r['partner_amount'] !== null): ?>
                                <div class="col-md-2">
                                  <div class="border rounded p-3 bg-white">
                                    <div class="text-muted small mb-1">
                                      <i class="bi bi-cash-stack me-1"></i>Amount Total
                                    </div>
                                    <div class="fw-bold fs-5 text-success">
                                      <?php echo number_format((float)$r['partner_amount'], 3); ?> <small class="text-muted">OMR</small>
                                    </div>
                                  </div>
                                </div>
                                <?php endif; ?>
                              </div>

                              <!-- Comparison Alert -->
                              <?php
                                $hasDiscrepancy = false;
                                $discrepancies = [];

                                if (isset($r['partner_orders']) && $r['partner_orders'] !== null && (int)$r['partner_orders'] !== (int)$r['orders_count']) {
                                  $hasDiscrepancy = true;
                                  $discrepancies[] = 'Orders count differs: Partner has ' . (int)$r['partner_orders'] . ', submission has ' . (int)$r['orders_count'];
                                }

                                if (isset($r['partner_amount']) && $r['partner_amount'] !== null) {
                                  $partnerAmt = (float)$r['partner_amount'];
                                  $submissionAmt = (float)$r['cash_deposited'];
                                  $diff = abs($partnerAmt - $submissionAmt);
                                  if ($diff > 0.001) { // Allow for small floating point differences
                                    $hasDiscrepancy = true;
                                    $discrepancies[] = 'Amount differs: Partner has ' . number_format($partnerAmt, 3) . ' OMR, submission has ' . number_format($submissionAmt, 3) . ' OMR';
                                  }
                                }
                              ?>

                              <?php if ($hasDiscrepancy): ?>
                                <div class="alert alert-warning mt-3 mb-0">
                                  <strong><i class="bi bi-exclamation-triangle me-2"></i>Data Mismatch Detected</strong>
                                  <ul class="mb-0 mt-2">
                                    <?php foreach ($discrepancies as $disc): ?>
                                      <li><?php echo e($disc); ?></li>
                                    <?php endforeach; ?>
                                  </ul>
                                </div>
                              <?php else: ?>
                                <div class="alert alert-success mt-3 mb-0">
                                  <i class="bi bi-check-circle me-2"></i>Partner data matches submission data
                                </div>
                              <?php endif; ?>
                            </div>
                          </div>
                        <?php endif; ?>

                        <!-- TRX Info -->
                        <?php if ($bankTrx !== '' || $driverTrx !== ''): ?>
                          <div class="info-badge mb-4 <?php echo $bankTrx !== '' ? 'border-success' : ''; ?>">
                            <i class="bi bi-<?php echo $bankTrx !== '' ? 'check-circle-fill text-success' : 'info-circle'; ?>"></i>
                            <div class="flex-grow-1">
                              <?php if ($bankTrx !== ''): ?>
                                <div class="fw-semibold text-success">Bank Transaction Matched</div>
                                <div class="trx-badge"><?php echo e($bankTrx); ?></div>
                                <?php if (!is_null($r['bank_amount']) || !is_null($r['bank_date'])): ?>
                                  <div class="text-muted small mt-1">
                                    <?php
                                      $bits = [];
                                      if (!is_null($r['bank_amount'])) $bits[] = 'OMR '.number_format((float)$r['bank_amount'],3);
                                      if (!is_null($r['bank_date']))   $bits[] = e($r['bank_date']);
                                      echo implode(' · ', $bits);
                                    ?>
                                  </div>
                                <?php endif; ?>
                                <?php if ($driverTrx !== '' && $driverTrx !== $bankTrx): ?>
                                  <div class="text-muted small">Driver entered: <?php echo e($driverTrx); ?></div>
                                <?php endif; ?>
                              <?php else: ?>
                                <div class="fw-semibold">Driver TRX Number</div>
                                <div class="trx-badge"><?php echo e($driverTrx); ?></div>
                                <div class="text-muted small">Not matched with bank records yet</div>
                              <?php endif; ?>
                            </div>
                          </div>
                        <?php endif; ?>

                        <!-- Edit Fields -->
                        <div class="card mb-4 shadow-sm">
                          <div class="card-header bg-white">
                            <h6 class="mb-0"><i class="bi bi-pencil-square"></i> Submission Details</h6>
                          </div>
                          <div class="card-body">
                            <div class="row g-3">
                              <div class="col-md-3">
                                <label class="form-label fw-semibold">Orders Count</label>
                                <input class="form-control form-control-lg" name="orders_count" type="number" min="0" value="<?php echo (int)$r['orders_count']; ?>" required>
                              </div>
                              <div class="col-md-3">
                                <label class="form-label fw-semibold">Cash Amount (OMR)</label>
                                <input class="form-control form-control-lg" name="cash_deposited" type="text" pattern="\d+(\.\d{1,3})?" value="<?php echo number_format((float)$r['cash_deposited'], 3, '.', ''); ?>" required>
                              </div>
                              <div class="col-md-3">
                                <label class="form-label fw-semibold">TRX Number</label>
                                <input class="form-control form-control-lg" name="trx_no" value="<?php echo e($driverTrx); ?>">
                              </div>
                              <div class="col-md-3">
                                <label class="form-label fw-semibold">Remarks</label>
                                <input class="form-control form-control-lg" name="remarks" value="<?php echo e($r['remarks'] ?? ''); ?>" maxlength="255">
                              </div>
                            </div>
                          </div>
                        </div>

                        <!-- Screenshots Grid -->
                        <div class="card shadow-sm">
                          <div class="card-header bg-white">
                            <h6 class="mb-0"><i class="bi bi-images"></i> Uploaded Screenshots</h6>
                          </div>
                          <div class="card-body p-4">
                            <div class="screenshot-grid">
                              <div class="screenshot-card">
                                <div class="screenshot-card-header">
                                  <i class="bi bi-card-list"></i> Order Screenshot
                                </div>
                                <div class="screenshot-card-body">
                                  <?php echo render_media_preview($r['ss_orders'] ?? null, false, true); ?>
                                </div>
                              </div>

                              <div class="screenshot-card">
                                <div class="screenshot-card-header">
                                  <i class="bi bi-receipt"></i> Deposit Slip 1
                                </div>
                                <div class="screenshot-card-body">
                                  <?php echo render_media_preview($r['ss_deposit_1'] ?? null, false, true); ?>
                                </div>
                              </div>

                              <div class="screenshot-card">
                                <div class="screenshot-card-header">
                                  <i class="bi bi-receipt"></i> Deposit Slip 2
                                </div>
                                <div class="screenshot-card-body">
                                  <?php echo render_media_preview($r['ss_deposit_2'] ?? null, false, true); ?>
                                </div>
                              </div>
                            </div>
                          </div>
                        </div>
                      </form>
                    </div>

                    <div class="modal-footer bg-white border-top">
                      <button type="button" class="btn btn-outline-secondary" data-bs-dismiss="modal">
                        <i class="bi bi-x-circle"></i> Cancel
                      </button>
                      <button form="formApprove_<?php echo $sid; ?>" name="action" value="update" class="btn btn-primary">
                        <i class="bi bi-save"></i> Save Changes
                      </button>
                      <button form="formApprove_<?php echo $sid; ?>" name="action" value="approve" class="btn btn-success btn-lg px-4">
                        <i class="bi bi-check-circle-fill"></i> Approve Submission
                      </button>
                    </div>
                  </div>
                </div>
              </div>

              <div class="modal fade" id="mReject_<?php echo $sid; ?>" tabindex="-1" aria-hidden="true">
                <div class="modal-dialog modal-md modal-dialog-centered"><div class="modal-content">
                  <div class="modal-header"><h6 class="modal-title">Reject submission</h6><button type="button" class="btn-close" data-bs-dismiss="modal"></button></div>
                  <form method="post">
                    <input type="hidden" name="csrf" value="<?php echo e($csrf); ?>">
                    <input type="hidden" name="submission_id" value="<?php echo $sid; ?>">
                    <div class="modal-body">
                      <label class="form-label">Reason</label>
                      <textarea class="form-control" name="reason" rows="3" placeholder="Optional"></textarea>
                    </div>
                    <div class="modal-footer">
                      <button class="btn btn-outline-secondary" type="button" data-bs-dismiss="modal">Cancel</button>
                      <button class="btn btn-danger" name="action" value="reject">Reject</button>
                    </div>
                  </form>
                </div></div>
              </div>
            <?php endforeach; endif; ?>
          </tbody>
        </table>
      </div>
    </div>
  </div>
</div>

<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>



