<?php
// public/admin/settings.php
require_once __DIR__ . '/../../includes/admin_bootstrap.php';
require_once __DIR__ . '/../../includes/uploads.php';

$role = get_current_role();
if (!can_edit_settings($role)) {
    http_response_code(403);
    exit('Forbidden');
}

$userId = current_user_id();
$settingsUrl = base_url('admin/settings.php');

initialize_settings_store($conn);

$flashOk  = flash('settings_ok');
$flashErr = flash('settings_err');
$csrf     = csrf_token();

// Branding setting keys
$brandingKeys = [
    'branding.app_name',
    'branding.admin_logo',
    'branding.admin_login_logo',
    'branding.driver_logo',
    'branding.driver_login_logo',
];

$backupDirs = ensure_backup_directories();
$backupDirError = $backupDirs ? '' : 'Backups directory is not writable. Check storage permissions.';

if (isset($_GET['backup_download']) && $_GET['backup_download'] === '1') {
    $type = trim((string)($_GET['type'] ?? ''));
    $file = trim((string)($_GET['file'] ?? ''));
    $path = resolve_backup_file($backupDirs, $type, $file);
    if (!$path) {
        http_response_code(404);
        exit('Backup not found.');
    }
    send_backup_file($path);
}

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $submittedCsrf = $_POST['csrf'] ?? '';
    if (!hash_equals($csrf, $submittedCsrf)) {
        flash('settings_err', 'Security token mismatch. Please try again.');
        redirect_to($settingsUrl);
    }

    $actionRaw   = trim((string)($_POST['action'] ?? ''));
    $deleteId    = null;
    $normalized  = $actionRaw;

    if (strpos($actionRaw, 'delete-') === 0) {
        $deleteId   = (int)substr($actionRaw, 7);
        $normalized = 'delete';
    }
    if (strpos($actionRaw, 'remove-logo-') === 0) {
        $normalized = 'remove-logo';
    }

    switch ($normalized) {
        case 'backup-create-database':
        case 'backup-create-codebase':
        case 'backup-create-storage':
            $backupType = str_replace('backup-create-', '', $normalized);
            if (!$backupDirs || !isset($backupDirs[$backupType])) {
                flash('settings_err', 'Backup directory is unavailable.');
                redirect_to($settingsUrl);
            }
            if ($backupType === 'database') {
                $result = create_database_backup($conn, $backupDirs['database']);
            } elseif ($backupType === 'codebase') {
                $rootDir = realpath(__DIR__ . '/../..');
                $result = create_codebase_backup($rootDir ?: '', $backupDirs['codebase']);
            } else {
                $result = create_storage_backup(STORAGE_DIR, $backupDirs['storage']);
            }
            if ($result['ok'] ?? false) {
                $fileName = $result['file'] ?? '';
                flash('settings_ok', $fileName ? "Backup created: {$fileName}" : 'Backup created.');
            } else {
                flash('settings_err', $result['message'] ?? 'Failed to create backup.');
            }
            redirect_to($settingsUrl);
            break;

        case 'backup-restore':
            $backupType = trim((string)($_POST['backup_type'] ?? ''));
            $backupFile = trim((string)($_POST['backup_file'] ?? ''));
            $backupPath = resolve_backup_file($backupDirs, $backupType, $backupFile);
            if (!$backupPath) {
                flash('settings_err', 'Backup file not found.');
                redirect_to($settingsUrl);
            }
            if ($backupType === 'database') {
                $result = restore_database_backup($conn, $backupPath);
            } elseif ($backupType === 'codebase') {
                $rootDir = realpath(__DIR__ . '/../..');
                $result = restore_codebase_backup($backupPath, $rootDir ?: '');
            } elseif ($backupType === 'storage') {
                $result = restore_storage_backup($backupPath, STORAGE_DIR);
            } else {
                $result = ['ok' => false, 'message' => 'Unsupported backup type.'];
            }
            if ($result['ok'] ?? false) {
                flash('settings_ok', $result['message'] ?? 'Restore completed.');
            } else {
                flash('settings_err', $result['message'] ?? 'Restore failed.');
            }
            redirect_to($settingsUrl);
            break;

        case 'backup-delete':
            $backupType = trim((string)($_POST['backup_type'] ?? ''));
            $backupFile = trim((string)($_POST['backup_file'] ?? ''));
            $backupPath = resolve_backup_file($backupDirs, $backupType, $backupFile);
            if (!$backupPath) {
                flash('settings_err', 'Backup file not found.');
                redirect_to($settingsUrl);
            }
            if (@unlink($backupPath)) {
                flash('settings_ok', 'Backup deleted.');
            } else {
                flash('settings_err', 'Unable to delete backup.');
            }
            redirect_to($settingsUrl);
            break;

        case 'save-branding':
            $updated = 0;

            // Save app name
            $appName = trim($_POST['branding_app_name'] ?? '');
            upsert_setting($conn, 'branding.app_name', $appName, 'Application name shown when no logo is set', $userId);
            $updated++;

            // Handle logo uploads
            $logoTypes = [
                'admin_logo' => 'branding.admin_logo',
                'admin_login_logo' => 'branding.admin_login_logo',
                'driver_logo' => 'branding.driver_logo',
                'driver_login_logo' => 'branding.driver_login_logo',
            ];

            foreach ($logoTypes as $field => $settingKey) {
                if (isset($_FILES[$field]) && $_FILES[$field]['error'] === UPLOAD_ERR_OK) {
                    $path = save_branding_file($_FILES[$field], $field);
                    if ($path !== '') {
                        upsert_setting($conn, $settingKey, $path, ucwords(str_replace('_', ' ', $field)), $userId);
                        $updated++;
                    } else {
                        flash('settings_err', 'Failed to upload ' . str_replace('_', ' ', $field) . '. Please use a valid image file.');
                        redirect_to($settingsUrl);
                    }
                }
            }

            invalidate_settings_cache();
            flash('settings_ok', 'Branding settings saved.');
            redirect_to($settingsUrl);
            break;

        case 'remove-logo':
            $logoType = str_replace('remove-logo-', '', $actionRaw);
            $validTypes = ['admin_logo', 'admin_login_logo', 'driver_logo', 'driver_login_logo'];
            if (in_array($logoType, $validTypes, true)) {
                delete_branding_file($logoType);
                upsert_setting($conn, 'branding.' . $logoType, '', ucwords(str_replace('_', ' ', $logoType)), $userId);
                invalidate_settings_cache();
                flash('settings_ok', ucwords(str_replace('_', ' ', $logoType)) . ' removed.');
            }
            redirect_to($settingsUrl);
            break;

        case 'save':
            $rows      = $_POST['rows'] ?? [];
            $updated   = 0;
            $hadError  = false;
            $errorMsg  = '';
            $keyLookup = fetch_setting_keys_by_ids($conn, array_keys($rows));

            if ($rows && $keyLookup) {
                $stmt = $conn->prepare("UPDATE app_settings SET setting_value=?, description=?, updated_by=?, updated_at=NOW() WHERE id=?");
                foreach ($rows as $id => $payload) {
                    $id = (int)$id;
                    if ($id <= 0 || !isset($keyLookup[$id])) {
                        continue;
                    }

                    $value = (string)($payload['value'] ?? '');
                    $desc  = trim((string)($payload['description'] ?? ''));
                    $key   = $keyLookup[$id];

                    if ($key === 'app.timezone' && !is_valid_timezone_identifier($value)) {
                        $hadError = true;
                        $errorMsg = 'Select a valid timezone from the dropdown.';
                        break;
                    }

                    $stmt->bind_param('ssii', $value, $desc, $userId, $id);
                    if (!$stmt->execute()) {
                        $hadError = true;
                        $errorMsg = 'Failed to update one or more settings.';
                        break;
                    }
                    $updated++;
                }
                $stmt->close();
            }

            invalidate_settings_cache();
            if ($hadError) {
                flash('settings_err', $errorMsg ?: 'Unable to save settings.');
            } else {
                flash('settings_ok', $updated ? "Saved {$updated} setting(s)." : 'No changes detected.');
            }
            redirect_to($settingsUrl);
            break;

        case 'create':
            $newKey   = strtolower(trim($_POST['new_key'] ?? ''));
            $newValue = (string)($_POST['new_value'] ?? '');
            $newDesc  = trim((string)($_POST['new_description'] ?? ''));

            if ($newKey === '' || !preg_match('/^[a-z0-9_.-]{3,}$/', $newKey)) {
                flash('settings_err', 'Provide a key using letters, numbers, dot, dash, or underscore (min 3 chars).');
                redirect_to($settingsUrl);
            }

            if ($newKey === 'app.timezone' && !is_valid_timezone_identifier($newValue)) {
                flash('settings_err', 'Select a valid timezone value.');
                redirect_to($settingsUrl);
            }

            $stmt = $conn->prepare("INSERT INTO app_settings (setting_key, setting_value, description, updated_by, updated_at)
                                    VALUES (?,?,?,?,NOW())");
            $stmt->bind_param('sssi', $newKey, $newValue, $newDesc, $userId);
            if ($stmt->execute()) {
                invalidate_settings_cache();
                flash('settings_ok', 'Setting created.');
            } else {
                $msg = $stmt->errno === 1062 ? 'Key already exists.' : 'Failed to create setting.';
                flash('settings_err', $msg);
            }
            $stmt->close();
            redirect_to($settingsUrl);
            break;

        case 'delete':
            if ($deleteId <= 0) {
                flash('settings_err', 'Invalid setting selected.');
                redirect_to($settingsUrl);
            }

            $stmt = $conn->prepare("DELETE FROM app_settings WHERE id=? LIMIT 1");
            $stmt->bind_param('i', $deleteId);
            $stmt->execute();
            $stmt->close();

            invalidate_settings_cache();
            flash('settings_ok', 'Setting deleted.');
            redirect_to($settingsUrl);
            break;

        default:
            flash('settings_err', 'Unsupported action.');
            redirect_to($settingsUrl);
    }
}

// Fetch all settings
$settings = [];
$sql = "
    SELECT s.id,
           s.setting_key,
           s.setting_value,
           s.description,
           s.updated_at,
           u.email AS updated_by_email
    FROM app_settings s
    LEFT JOIN users u ON u.id = s.updated_by
    ORDER BY s.setting_key ASC
";
if ($result = $conn->query($sql)) {
    $settings = $result->fetch_all(MYSQLI_ASSOC);
    $result->free();
}

// Build settings lookup
$settingsMap = [];
foreach ($settings as $s) {
    $settingsMap[$s['setting_key']] = $s;
}

$timezoneOptions = timezone_options();

// Group settings by category
$groupedSettings = [
    'general' => [],
    'other' => []
];

foreach ($settings as $setting) {
    $key = $setting['setting_key'];
    if ($key === 'app.timezone') {
        $groupedSettings['general'][] = $setting;
    } elseif (!in_array($key, $brandingKeys, true) && $key !== 'admin.header_logo') {
        $groupedSettings['other'][] = $setting;
    }
}

// Get current branding values
$brandingAppName = $settingsMap['branding.app_name']['setting_value'] ?? 'Hisbuu';
$brandingAdminLogo = $settingsMap['branding.admin_logo']['setting_value'] ?? '';
$brandingAdminLoginLogo = $settingsMap['branding.admin_login_logo']['setting_value'] ?? '';
$brandingDriverLogo = $settingsMap['branding.driver_logo']['setting_value'] ?? '';
$brandingDriverLoginLogo = $settingsMap['branding.driver_login_logo']['setting_value'] ?? '';

$backupItems = $backupDirs ? backup_inventory($backupDirs) : [];
$backupTypeLabels = [
    'database' => 'Database',
    'codebase' => 'Codebase',
    'storage' => 'Storage',
];
$backupDisabled = $backupDirs ? '' : 'disabled';

$PAGE_TITLE = 'System Settings';
@include __DIR__ . '/../../includes/admin_header.php';
?>
<div class="d-flex justify-content-between align-items-center mb-3">
  <div>
    <h5 class="mb-0">System Settings</h5>
  </div>
</div>

<?php if ($flashOk): ?>
  <div class="alert alert-success"><?php echo e($flashOk); ?></div>
<?php endif; ?>
<?php if ($flashErr): ?>
  <div class="alert alert-danger"><?php echo e($flashErr); ?></div>
<?php endif; ?>

<!-- Branding Section -->
<form method="post" enctype="multipart/form-data" class="mb-4">
  <?php echo csrf_input(); ?>
  <input type="hidden" name="action" value="save-branding">

  <div class="card mb-3">
    <div class="card-header bg-light">
      <h6 class="mb-0"><i class="bi bi-palette me-2"></i>Branding</h6>
    </div>
    <div class="card-body">

      <!-- App Name -->
      <div class="mb-4">
        <label class="form-label fw-semibold">App Name</label>
        <input type="text" class="form-control" name="branding_app_name"
               value="<?php echo e($brandingAppName); ?>"
               placeholder="Hisbuu" maxlength="50">
        <div class="form-text">Displayed when no logo is set. Also used for page titles.</div>
      </div>

      <hr class="my-4">

      <!-- Logo Uploads Grid -->
      <div class="row g-4">
        <!-- Admin Panel Logo -->
        <div class="col-md-6">
          <div class="border rounded p-3 h-100">
            <label class="form-label fw-semibold">Admin Panel Logo</label>
            <div class="form-text mb-2">Shown in admin sidebar and top bar. Recommended: 180x30px, PNG/SVG with transparent background.</div>

            <div class="d-flex align-items-center gap-3 mb-2">
              <div class="border rounded bg-light d-flex align-items-center justify-content-center" style="width:80px;height:50px;">
                <?php if ($brandingAdminLogo): ?>
                  <img src="<?php echo e(file_url($brandingAdminLogo)); ?>" alt="Admin Logo" style="max-height:45px;max-width:75px;object-fit:contain;">
                <?php else: ?>
                  <div class="text-muted small">No logo</div>
                <?php endif; ?>
              </div>
              <div class="flex-grow-1">
                <input type="file" class="form-control form-control-sm" name="admin_logo" accept="image/*">
              </div>
            </div>
            <?php if ($brandingAdminLogo): ?>
              <button type="submit" name="action" value="remove-logo-admin_logo" class="btn btn-outline-danger btn-sm">
                <i class="bi bi-trash"></i> Remove
              </button>
            <?php endif; ?>
          </div>
        </div>

        <!-- Admin Login Logo -->
        <div class="col-md-6">
          <div class="border rounded p-3 h-100">
            <label class="form-label fw-semibold">Admin Login Logo</label>
            <div class="form-text mb-2">Shown on admin login page. Recommended: 180x40px, PNG/SVG with transparent background.</div>

            <div class="d-flex align-items-center gap-3 mb-2">
              <div class="border rounded bg-light d-flex align-items-center justify-content-center" style="width:80px;height:50px;">
                <?php if ($brandingAdminLoginLogo): ?>
                  <img src="<?php echo e(file_url($brandingAdminLoginLogo)); ?>" alt="Admin Login Logo" style="max-height:45px;max-width:75px;object-fit:contain;">
                <?php else: ?>
                  <div class="text-muted small">No logo</div>
                <?php endif; ?>
              </div>
              <div class="flex-grow-1">
                <input type="file" class="form-control form-control-sm" name="admin_login_logo" accept="image/*">
              </div>
            </div>
            <?php if ($brandingAdminLoginLogo): ?>
              <button type="submit" name="action" value="remove-logo-admin_login_logo" class="btn btn-outline-danger btn-sm">
                <i class="bi bi-trash"></i> Remove
              </button>
            <?php endif; ?>
          </div>
        </div>

        <!-- Driver Portal Logo -->
        <div class="col-md-6">
          <div class="border rounded p-3 h-100">
            <label class="form-label fw-semibold">Driver Portal Logo</label>
            <div class="form-text mb-2">Shown in driver portal header. Recommended: 140x36px, PNG/SVG with transparent background.</div>

            <div class="d-flex align-items-center gap-3 mb-2">
              <div class="border rounded bg-light d-flex align-items-center justify-content-center" style="width:80px;height:50px;">
                <?php if ($brandingDriverLogo): ?>
                  <img src="<?php echo e(file_url($brandingDriverLogo)); ?>" alt="Driver Logo" style="max-height:45px;max-width:75px;object-fit:contain;">
                <?php else: ?>
                  <div class="text-muted small">No logo</div>
                <?php endif; ?>
              </div>
              <div class="flex-grow-1">
                <input type="file" class="form-control form-control-sm" name="driver_logo" accept="image/*">
              </div>
            </div>
            <?php if ($brandingDriverLogo): ?>
              <button type="submit" name="action" value="remove-logo-driver_logo" class="btn btn-outline-danger btn-sm">
                <i class="bi bi-trash"></i> Remove
              </button>
            <?php endif; ?>
          </div>
        </div>

        <!-- Driver Login Logo -->
        <div class="col-md-6">
          <div class="border rounded p-3 h-100">
            <label class="form-label fw-semibold">Driver Login Logo</label>
            <div class="form-text mb-2">Shown on driver login page. Recommended: 180x40px, PNG/SVG with transparent background.</div>

            <div class="d-flex align-items-center gap-3 mb-2">
              <div class="border rounded bg-light d-flex align-items-center justify-content-center" style="width:80px;height:50px;">
                <?php if ($brandingDriverLoginLogo): ?>
                  <img src="<?php echo e(file_url($brandingDriverLoginLogo)); ?>" alt="Driver Login Logo" style="max-height:45px;max-width:75px;object-fit:contain;">
                <?php else: ?>
                  <div class="text-muted small">No logo</div>
                <?php endif; ?>
              </div>
              <div class="flex-grow-1">
                <input type="file" class="form-control form-control-sm" name="driver_login_logo" accept="image/*">
              </div>
            </div>
            <?php if ($brandingDriverLoginLogo): ?>
              <button type="submit" name="action" value="remove-logo-driver_login_logo" class="btn btn-outline-danger btn-sm">
                <i class="bi bi-trash"></i> Remove
              </button>
            <?php endif; ?>
          </div>
        </div>
      </div>

      <div class="mt-4">
        <button type="submit" class="btn btn-primary">
          <i class="bi bi-check-lg me-1"></i> Save Branding
        </button>
      </div>
    </div>
  </div>
</form>

<?php if ($backupDirError): ?>
  <div class="alert alert-warning"><?php echo e($backupDirError); ?></div>
<?php endif; ?>

<div class="card mb-4">
  <div class="card-header bg-light">
    <h6 class="mb-0"><i class="bi bi-shield-check me-2"></i>Backups & Restore</h6>
  </div>
  <div class="card-body">
    <div class="text-muted small mb-3">
      Create separate backups for database, codebase, and storage. Restores overwrite current data.
    </div>
    <div class="row g-3">
      <div class="col-md-4">
        <div class="border rounded p-3 h-100">
          <div class="fw-semibold mb-1">Database</div>
          <div class="text-muted small mb-3">SQL dump of all tables.</div>
          <form method="post">
            <?php echo csrf_input(); ?>
            <input type="hidden" name="action" value="backup-create-database">
            <button type="submit" class="btn btn-outline-primary btn-sm" <?php echo $backupDisabled; ?>>
              <i class="bi bi-database me-1"></i> Create backup
            </button>
          </form>
        </div>
      </div>
      <div class="col-md-4">
        <div class="border rounded p-3 h-100">
          <div class="fw-semibold mb-1">Codebase</div>
          <div class="text-muted small mb-3">App files excluding storage.</div>
          <form method="post">
            <?php echo csrf_input(); ?>
            <input type="hidden" name="action" value="backup-create-codebase">
            <button type="submit" class="btn btn-outline-primary btn-sm" <?php echo $backupDisabled; ?>>
              <i class="bi bi-archive me-1"></i> Create backup
            </button>
          </form>
        </div>
      </div>
      <div class="col-md-4">
        <div class="border rounded p-3 h-100">
          <div class="fw-semibold mb-1">Storage</div>
          <div class="text-muted small mb-3">Uploads, submissions, and branding.</div>
          <form method="post">
            <?php echo csrf_input(); ?>
            <input type="hidden" name="action" value="backup-create-storage">
            <button type="submit" class="btn btn-outline-primary btn-sm" <?php echo $backupDisabled; ?>>
              <i class="bi bi-folder2-open me-1"></i> Create backup
            </button>
          </form>
        </div>
      </div>
    </div>

    <hr class="my-4">

    <div class="fw-semibold mb-2">Existing backups</div>
    <?php if (empty($backupItems)): ?>
      <div class="text-muted small">No backups created yet.</div>
    <?php else: ?>
      <div class="table-responsive">
        <table class="table table-striped align-middle mb-0">
          <thead>
            <tr>
              <th>Type</th>
              <th>File</th>
              <th class="text-end">Size</th>
              <th>Created</th>
              <th class="text-end">Actions</th>
            </tr>
          </thead>
          <tbody>
            <?php foreach ($backupItems as $item): ?>
              <tr>
                <td><?php echo e($backupTypeLabels[$item['type']] ?? $item['type']); ?></td>
                <td class="text-break"><?php echo e($item['name']); ?></td>
                <td class="text-end"><?php echo e(format_bytes($item['size'])); ?></td>
                <td><?php echo e(date('d M Y H:i', $item['time'])); ?></td>
                <td class="text-end">
                  <a class="btn btn-outline-secondary btn-sm"
                     href="<?php echo $settingsUrl; ?>?backup_download=1&type=<?php echo urlencode($item['type']); ?>&file=<?php echo rawurlencode($item['name']); ?>">
                    Download
                  </a>
                  <form method="post" class="d-inline">
                    <?php echo csrf_input(); ?>
                    <input type="hidden" name="action" value="backup-restore">
                    <input type="hidden" name="backup_type" value="<?php echo e($item['type']); ?>">
                    <input type="hidden" name="backup_file" value="<?php echo e($item['name']); ?>">
                    <button type="submit" class="btn btn-outline-warning btn-sm" <?php echo $backupDisabled; ?>
                            onclick="return confirm('Restore <?php echo e($backupTypeLabels[$item['type']] ?? $item['type']); ?> backup? This will overwrite current data.');">
                      Restore
                    </button>
                  </form>
                  <form method="post" class="d-inline">
                    <?php echo csrf_input(); ?>
                    <input type="hidden" name="action" value="backup-delete">
                    <input type="hidden" name="backup_type" value="<?php echo e($item['type']); ?>">
                    <input type="hidden" name="backup_file" value="<?php echo e($item['name']); ?>">
                    <button type="submit" class="btn btn-outline-danger btn-sm" <?php echo $backupDisabled; ?>
                            onclick="return confirm('Delete this backup file?');">
                      Delete
                    </button>
                  </form>
                </td>
              </tr>
            <?php endforeach; ?>
          </tbody>
        </table>
      </div>
    <?php endif; ?>
    <div class="text-muted small mt-2">Large backups may take several minutes to create or restore.</div>
  </div>
</div>

<form method="post">
  <?php echo csrf_input(); ?>

  <!-- General Settings -->
  <?php if (!empty($groupedSettings['general'])): ?>
  <div class="card mb-3">
    <div class="card-header bg-light">
      <h6 class="mb-0"><i class="bi bi-gear me-2"></i>General</h6>
    </div>
    <div class="card-body">
      <?php foreach ($groupedSettings['general'] as $setting): ?>
        <div class="mb-3">
          <label class="form-label fw-semibold">
            <?php echo e($setting['setting_key']); ?>
            <span class="badge bg-info-subtle text-info-emphasis ms-2">Timezone</span>
          </label>
          <select
            class="form-select"
            name="rows[<?php echo (int)$setting['id']; ?>][value]"
          >
            <?php foreach ($timezoneOptions as $tz): ?>
              <option value="<?php echo e($tz); ?>" <?php echo $setting['setting_value'] === $tz ? 'selected' : ''; ?>>
                <?php echo e($tz); ?>
              </option>
            <?php endforeach; ?>
          </select>
          <div class="form-text small text-muted">All PHP + MySQL timestamps will use this timezone.</div>
          <input type="hidden" name="rows[<?php echo (int)$setting['id']; ?>][description]" value="<?php echo e($setting['description']); ?>">
        </div>
      <?php endforeach; ?>
    </div>
  </div>
  <?php endif; ?>

  <!-- Other Settings -->
  <?php if (!empty($groupedSettings['other'])): ?>
  <div class="card mb-3">
    <div class="card-header bg-light">
      <h6 class="mb-0"><i class="bi bi-list-ul me-2"></i>Other Settings</h6>
    </div>
    <div class="card-body">
      <?php foreach ($groupedSettings['other'] as $setting): ?>
        <div class="mb-3">
          <label class="form-label fw-semibold">
            <?php echo e($setting['setting_key']); ?>
          </label>
          <textarea
            class="form-control"
            rows="3"
            name="rows[<?php echo (int)$setting['id']; ?>][value]"
            spellcheck="false"
            placeholder="Enter value"
          ><?php echo e($setting['setting_value']); ?></textarea>
          <?php if ($setting['description']): ?>
            <div class="form-text small text-muted"><?php echo e($setting['description']); ?></div>
          <?php endif; ?>
          <input type="hidden" name="rows[<?php echo (int)$setting['id']; ?>][description]" value="<?php echo e($setting['description']); ?>">
          <div class="d-flex justify-content-between align-items-center mt-2">
            <small class="text-muted">
              Updated: <?php echo $setting['updated_at'] ? e(format_date($setting['updated_at'])) : '—'; ?>
              <?php if (!empty($setting['updated_by_email'])): ?>
                · <?php echo e($setting['updated_by_email']); ?>
              <?php endif; ?>
            </small>
            <button
              type="submit"
              name="action"
              value="delete-<?php echo (int)$setting['id']; ?>"
              class="btn btn-outline-danger btn-sm"
              onclick="return confirm('Delete <?php echo e($setting['setting_key']); ?>?');"
            >
              <i class="bi bi-trash"></i> Delete
            </button>
          </div>
        </div>
      <?php endforeach; ?>
    </div>
  </div>
  <?php endif; ?>

  <!-- Save Button -->
  <?php if (!empty($groupedSettings['general']) || !empty($groupedSettings['other'])): ?>
  <div class="d-flex justify-content-end gap-2 mb-4">
    <button type="submit" name="action" value="save" class="btn btn-primary">
      <i class="bi bi-check-lg me-1"></i> Save Changes
    </button>
  </div>
  <?php endif; ?>
</form>

<!-- Add New Setting -->
<div class="card">
  <div class="card-header">
    <h6 class="mb-0"><i class="bi bi-plus-circle me-2"></i>Add New Setting</h6>
  </div>
  <div class="card-body">
    <form method="post" class="row g-3">
      <?php echo csrf_input(); ?>
      <input type="hidden" name="action" value="create">
      <div class="col-md-4">
        <label class="form-label fw-semibold">Key</label>
        <input class="form-control" type="text" name="new_key" placeholder="e.g. partner.cutoff_hour">
        <div class="form-text small text-muted">Lowercase letters, digits, dot, dash, underscore (min 3 chars).</div>
      </div>
      <div class="col-md-4">
        <label class="form-label fw-semibold">Value</label>
        <textarea class="form-control" name="new_value" rows="2" placeholder="Enter value"></textarea>
      </div>
      <div class="col-md-4">
        <label class="form-label fw-semibold">Description</label>
        <input class="form-control" type="text" name="new_description" placeholder="How this setting is used">
      </div>
      <div class="col-12 text-end">
        <button class="btn btn-success" type="submit">
          <i class="bi bi-plus-lg me-1"></i> Create Setting
        </button>
      </div>
    </form>
  </div>
</div>

<?php @include __DIR__ . '/../../includes/admin_footer.php'; ?>

<?php
function initialize_settings_store(mysqli $conn): void {
    $sql = <<<SQL
CREATE TABLE IF NOT EXISTS app_settings (
    id INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    setting_key VARCHAR(191) NOT NULL UNIQUE,
    setting_value TEXT NULL,
    description VARCHAR(255) DEFAULT NULL,
    updated_by INT DEFAULT NULL,
    updated_at TIMESTAMP NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    INDEX idx_setting_key (setting_key)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;
SQL;

    if (!$conn->query($sql)) {
        http_response_code(500);
        exit('Unable to initialize settings storage: ' . $conn->error);
    }

    // Insert default settings
    $defaults = [
        ['app.timezone', app_default_timezone(), 'PHP + MySQL default timezone'],
        ['branding.app_name', 'Hisbuu', 'Application name shown when no logo is set'],
        ['branding.admin_logo', '', 'Admin panel header logo'],
        ['branding.admin_login_logo', '', 'Admin login page logo'],
        ['branding.driver_logo', '', 'Driver portal header logo'],
        ['branding.driver_login_logo', '', 'Driver login page logo'],
    ];

    $stmt = $conn->prepare("INSERT IGNORE INTO app_settings (setting_key, setting_value, description, updated_at) VALUES (?, ?, ?, NOW())");
    foreach ($defaults as [$key, $value, $desc]) {
        $stmt->bind_param('sss', $key, $value, $desc);
        $stmt->execute();
    }
    $stmt->close();
}

function upsert_setting(mysqli $conn, string $key, string $value, string $desc, int $userId): bool {
    $stmt = $conn->prepare("INSERT INTO app_settings (setting_key, setting_value, description, updated_by, updated_at)
                            VALUES (?, ?, ?, ?, NOW())
                            ON DUPLICATE KEY UPDATE setting_value=VALUES(setting_value), description=VALUES(description), updated_by=VALUES(updated_by), updated_at=NOW()");
    $stmt->bind_param('sssi', $key, $value, $desc, $userId);
    $result = $stmt->execute();
    $stmt->close();
    return $result;
}

function fetch_setting_keys_by_ids(mysqli $conn, array $ids): array {
    $cleanIds = [];
    foreach ($ids as $id) {
        $id = (int)$id;
        if ($id > 0) {
            $cleanIds[] = $id;
        }
    }
    if (!$cleanIds) {
        return [];
    }

    $idList = implode(',', $cleanIds);
    $result = $conn->query("SELECT id, setting_key FROM app_settings WHERE id IN ($idList)");
    if (!$result) {
        return [];
    }

    $map = [];
    while ($row = $result->fetch_assoc()) {
        $map[(int)$row['id']] = $row['setting_key'];
    }
    $result->free();
    return $map;
}

function timezone_options(): array {
    static $cached = null;
    if ($cached !== null) {
        return $cached;
    }
    $cached = DateTimeZone::listIdentifiers();
    return $cached;
}

function is_valid_timezone_identifier(string $tz): bool {
    if ($tz === '') {
        return false;
    }
    $options = timezone_options();
    return in_array($tz, $options, true);
}

function app_default_timezone(): string {
    return 'Asia/Muscat';
}

function ensure_backup_directories(): array {
    if (!defined('STORAGE_DIR')) {
        return [];
    }
    $base = rtrim(STORAGE_DIR, DIRECTORY_SEPARATOR) . DIRECTORY_SEPARATOR . 'backups';
    $dirs = [
        'database' => $base . DIRECTORY_SEPARATOR . 'database',
        'codebase' => $base . DIRECTORY_SEPARATOR . 'codebase',
        'storage' => $base . DIRECTORY_SEPARATOR . 'storage',
    ];
    foreach ($dirs as $dir) {
        if (!is_dir($dir)) {
            if (!mkdir($dir, 0775, true) && !is_dir($dir)) {
                return [];
            }
        }
    }
    return $dirs;
}

function backup_inventory(array $dirs): array {
    $items = [];
    foreach ($dirs as $type => $dir) {
        if (!is_dir($dir)) {
            continue;
        }
        $paths = glob($dir . DIRECTORY_SEPARATOR . '*');
        if (!$paths) {
            continue;
        }
        foreach ($paths as $path) {
            if (!is_file($path)) {
                continue;
            }
            $items[] = [
                'type' => $type,
                'name' => basename($path),
                'size' => filesize($path) ?: 0,
                'time' => filemtime($path) ?: 0,
            ];
        }
    }
    usort($items, function ($a, $b) {
        return ($b['time'] ?? 0) <=> ($a['time'] ?? 0);
    });
    return $items;
}

function resolve_backup_file(array $dirs, string $type, string $name): ?string {
    if (!isset($dirs[$type])) {
        return null;
    }
    $name = basename($name);
    if ($name === '' || preg_match('/[^A-Za-z0-9._-]/', $name)) {
        return null;
    }
    $path = $dirs[$type] . DIRECTORY_SEPARATOR . $name;
    if (!is_file($path)) {
        return null;
    }
    return $path;
}

function format_bytes(?int $bytes): string {
    if ($bytes === null) {
        return '-';
    }
    $units = ['B', 'KB', 'MB', 'GB', 'TB'];
    $size = (float)$bytes;
    $i = 0;
    while ($size >= 1024 && $i < count($units) - 1) {
        $size /= 1024;
        $i++;
    }
    if ($i === 0) {
        return $bytes . ' ' . $units[$i];
    }
    return number_format($size, 1) . ' ' . $units[$i];
}

function send_backup_file(string $path): void {
    if (!is_file($path) || !is_readable($path)) {
        http_response_code(404);
        exit('File not found.');
    }
    $finfo = finfo_open(FILEINFO_MIME_TYPE);
    $mime = $finfo ? finfo_file($finfo, $path) : 'application/octet-stream';
    if ($finfo) {
        finfo_close($finfo);
    }
    header('Content-Type: ' . ($mime ?: 'application/octet-stream'));
    header('Content-Length: ' . filesize($path));
    header('Content-Disposition: attachment; filename="' . basename($path) . '"');
    header('Cache-Control: private, max-age=0');
    readfile($path);
    exit;
}

function should_skip_path(string $relative, array $skipPrefixes): bool {
    $relative = ltrim(str_replace('\\', '/', $relative), '/');
    if ($relative === '') {
        return true;
    }
    foreach ($skipPrefixes as $prefix) {
        $prefix = trim((string)$prefix, '/');
        if ($prefix === '') {
            continue;
        }
        if ($relative === $prefix || strpos($relative, $prefix . '/') === 0) {
            return true;
        }
    }
    return false;
}

function create_database_backup(mysqli $conn, string $destDir): array {
    @set_time_limit(0);
    if ($destDir === '' || !is_dir($destDir) || !is_writable($destDir)) {
        return ['ok' => false, 'message' => 'Backup directory is not writable.'];
    }
    $file = $destDir . DIRECTORY_SEPARATOR . 'database-' . date('Ymd-His') . '.sql';
    $fp = fopen($file, 'wb');
    if (!$fp) {
        return ['ok' => false, 'message' => 'Unable to create backup file.'];
    }
    fwrite($fp, "-- Database backup\n");
    fwrite($fp, "-- Generated: " . date('c') . "\n\n");
    fwrite($fp, "SET FOREIGN_KEY_CHECKS=0;\n\n");

    $tablesRes = $conn->query('SHOW TABLES');
    if (!$tablesRes) {
        fclose($fp);
        return ['ok' => false, 'message' => 'Unable to read database tables.'];
    }
    while ($row = $tablesRes->fetch_array(MYSQLI_NUM)) {
        $table = $row[0];
        if ($table === '') {
            continue;
        }
        $createRes = $conn->query("SHOW CREATE TABLE `{$table}`");
        if ($createRes) {
            $createRow = $createRes->fetch_assoc();
            $createRes->free();
            $createSql = $createRow['Create Table'] ?? '';
            if ($createSql !== '') {
                fwrite($fp, "-- Table `{$table}`\n");
                fwrite($fp, "DROP TABLE IF EXISTS `{$table}`;\n");
                fwrite($fp, $createSql . ";\n\n");
            }
        }

        $dataRes = $conn->query("SELECT * FROM `{$table}`", MYSQLI_USE_RESULT);
        if ($dataRes instanceof mysqli_result) {
            $fields = $dataRes->fetch_fields();
            $colNames = [];
            foreach ($fields as $field) {
                $colNames[] = '`' . $field->name . '`';
            }
            $insertPrefix = "INSERT INTO `{$table}` (" . implode(',', $colNames) . ") VALUES ";
            $batch = [];
            while ($dataRow = $dataRes->fetch_assoc()) {
                $values = [];
                foreach ($fields as $field) {
                    $val = $dataRow[$field->name];
                    if ($val === null) {
                        $values[] = 'NULL';
                    } else {
                        $values[] = "'" . $conn->real_escape_string($val) . "'";
                    }
                }
                $batch[] = '(' . implode(',', $values) . ')';
                if (count($batch) >= 500) {
                    fwrite($fp, $insertPrefix . implode(",\n", $batch) . ";\n");
                    $batch = [];
                }
            }
            if ($batch) {
                fwrite($fp, $insertPrefix . implode(",\n", $batch) . ";\n\n");
            }
            $dataRes->free();
        }
    }
    $tablesRes->free();
    fwrite($fp, "\nSET FOREIGN_KEY_CHECKS=1;\n");
    fclose($fp);
    return ['ok' => true, 'file' => basename($file)];
}

function create_codebase_backup(string $rootDir, string $destDir): array {
    @set_time_limit(0);
    if ($rootDir === '' || !is_dir($rootDir)) {
        return ['ok' => false, 'message' => 'Invalid codebase directory.'];
    }
    if ($destDir === '' || !is_dir($destDir) || !is_writable($destDir)) {
        return ['ok' => false, 'message' => 'Backup directory is not writable.'];
    }
    if (!class_exists('ZipArchive')) {
        return ['ok' => false, 'message' => 'ZipArchive extension is not available.'];
    }
    $file = $destDir . DIRECTORY_SEPARATOR . 'codebase-' . date('Ymd-His') . '.zip';
    $zip = new ZipArchive();
    if ($zip->open($file, ZipArchive::CREATE | ZipArchive::OVERWRITE) !== true) {
        return ['ok' => false, 'message' => 'Unable to create codebase archive.'];
    }

    $exclude = ['storage', '.git'];
    $iterator = new RecursiveIteratorIterator(
        new RecursiveDirectoryIterator($rootDir, FilesystemIterator::SKIP_DOTS)
    );
    foreach ($iterator as $fileInfo) {
        if (!$fileInfo->isFile()) {
            continue;
        }
        $path = $fileInfo->getPathname();
        $relative = substr($path, strlen($rootDir) + 1);
        $relative = str_replace('\\', '/', $relative);
        if (should_skip_path($relative, $exclude)) {
            continue;
        }
        $zip->addFile($path, $relative);
    }
    $zip->close();
    return ['ok' => true, 'file' => basename($file)];
}

function create_storage_backup(string $storageDir, string $destDir): array {
    @set_time_limit(0);
    if ($storageDir === '' || !is_dir($storageDir)) {
        return ['ok' => false, 'message' => 'Storage directory is missing.'];
    }
    if ($destDir === '' || !is_dir($destDir) || !is_writable($destDir)) {
        return ['ok' => false, 'message' => 'Backup directory is not writable.'];
    }
    if (!class_exists('ZipArchive')) {
        return ['ok' => false, 'message' => 'ZipArchive extension is not available.'];
    }
    $file = $destDir . DIRECTORY_SEPARATOR . 'storage-' . date('Ymd-His') . '.zip';
    $zip = new ZipArchive();
    if ($zip->open($file, ZipArchive::CREATE | ZipArchive::OVERWRITE) !== true) {
        return ['ok' => false, 'message' => 'Unable to create storage archive.'];
    }

    $iterator = new RecursiveIteratorIterator(
        new RecursiveDirectoryIterator($storageDir, FilesystemIterator::SKIP_DOTS)
    );
    foreach ($iterator as $fileInfo) {
        if (!$fileInfo->isFile()) {
            continue;
        }
        $path = $fileInfo->getPathname();
        $relative = substr($path, strlen($storageDir) + 1);
        $relative = str_replace('\\', '/', $relative);
        if (should_skip_path($relative, ['backups'])) {
            continue;
        }
        $zip->addFile($path, 'storage/' . $relative);
    }
    $zip->close();
    return ['ok' => true, 'file' => basename($file)];
}

function restore_database_backup(mysqli $conn, string $path): array {
    @set_time_limit(0);
    if (!is_readable($path)) {
        return ['ok' => false, 'message' => 'Backup file is not readable.'];
    }
    $sql = file_get_contents($path);
    if ($sql === false) {
        return ['ok' => false, 'message' => 'Unable to read backup file.'];
    }
    $conn->query('SET FOREIGN_KEY_CHECKS=0');
    if (!$conn->multi_query($sql)) {
        $conn->query('SET FOREIGN_KEY_CHECKS=1');
        return ['ok' => false, 'message' => 'Database restore failed: ' . $conn->error];
    }
    do {
        if ($result = $conn->store_result()) {
            $result->free();
        }
    } while ($conn->more_results() && $conn->next_result());
    $conn->query('SET FOREIGN_KEY_CHECKS=1');
    if ($conn->error) {
        return ['ok' => false, 'message' => 'Database restore failed: ' . $conn->error];
    }
    return ['ok' => true, 'message' => 'Database restored successfully.'];
}

function restore_codebase_backup(string $zipPath, string $rootDir): array {
    @set_time_limit(0);
    if ($rootDir === '' || !is_dir($rootDir)) {
        return ['ok' => false, 'message' => 'Invalid restore destination.'];
    }
    return extract_zip_safe($zipPath, $rootDir, null, ['storage', '.git']);
}

function restore_storage_backup(string $zipPath, string $storageDir): array {
    @set_time_limit(0);
    if ($storageDir === '' || !is_dir($storageDir)) {
        return ['ok' => false, 'message' => 'Invalid storage destination.'];
    }
    if (!class_exists('ZipArchive')) {
        return ['ok' => false, 'message' => 'ZipArchive extension is not available.'];
    }
    $zip = new ZipArchive();
    if ($zip->open($zipPath) !== true) {
        return ['ok' => false, 'message' => 'Unable to open storage backup.'];
    }
    $stripPrefix = zip_has_prefix($zip, 'storage/') ? 'storage/' : null;
    $zip->close();
    return extract_zip_safe($zipPath, $storageDir, $stripPrefix, ['backups']);
}

function extract_zip_safe(string $zipPath, string $destRoot, ?string $stripPrefix, array $skipPrefixes): array {
    if (!class_exists('ZipArchive')) {
        return ['ok' => false, 'message' => 'ZipArchive extension is not available.'];
    }
    if (!is_readable($zipPath)) {
        return ['ok' => false, 'message' => 'Backup file is not readable.'];
    }
    $zip = new ZipArchive();
    if ($zip->open($zipPath) !== true) {
        return ['ok' => false, 'message' => 'Unable to open backup archive.'];
    }
    $destRootReal = realpath($destRoot);
    if ($destRootReal === false) {
        $zip->close();
        return ['ok' => false, 'message' => 'Restore destination is invalid.'];
    }
    $destRootReal = rtrim($destRootReal, DIRECTORY_SEPARATOR) . DIRECTORY_SEPARATOR;
    $stripPrefix = $stripPrefix !== null ? ltrim(str_replace('\\', '/', $stripPrefix), '/') : null;

    for ($i = 0; $i < $zip->numFiles; $i++) {
        $entryName = $zip->getNameIndex($i);
        if ($entryName === false) {
            continue;
        }
        $normalized = str_replace('\\', '/', $entryName);
        if ($normalized === '' || substr($normalized, -1) === '/') {
            continue;
        }
        if (strpos($normalized, "\0") !== false) {
            continue;
        }
        if (preg_match('/(^[\\/])|(\\.{2})|(^[A-Za-z]:)/', $normalized)) {
            continue;
        }
        if ($stripPrefix !== null) {
            if (strpos($normalized, $stripPrefix) !== 0) {
                continue;
            }
            $normalized = substr($normalized, strlen($stripPrefix));
        }
        $normalized = ltrim($normalized, '/');
        if ($normalized === '') {
            continue;
        }
        if (should_skip_path($normalized, $skipPrefixes)) {
            continue;
        }
        $destPath = $destRootReal . str_replace('/', DIRECTORY_SEPARATOR, $normalized);
        $destDir = dirname($destPath);
        if (!is_dir($destDir)) {
            if (!mkdir($destDir, 0775, true) && !is_dir($destDir)) {
                $zip->close();
                return ['ok' => false, 'message' => 'Unable to create restore directories.'];
            }
        }
        $stream = $zip->getStream($entryName);
        if (!$stream) {
            $zip->close();
            return ['ok' => false, 'message' => 'Unable to read backup archive.'];
        }
        $out = fopen($destPath, 'wb');
        if (!$out) {
            fclose($stream);
            $zip->close();
            return ['ok' => false, 'message' => 'Unable to write restore file.'];
        }
        while (!feof($stream)) {
            $buffer = fread($stream, 8192);
            if ($buffer === false) {
                break;
            }
            fwrite($out, $buffer);
        }
        fclose($stream);
        fclose($out);
    }
    $zip->close();
    return ['ok' => true, 'message' => 'Restore completed.'];
}

function zip_has_prefix($zip, string $prefix): bool {
    $prefix = ltrim(str_replace('\\', '/', $prefix), '/');
    for ($i = 0; $i < $zip->numFiles; $i++) {
        $name = $zip->getNameIndex($i);
        if ($name === false) {
            continue;
        }
        $name = ltrim(str_replace('\\', '/', $name), '/');
        if (strpos($name, $prefix) === 0) {
            return true;
        }
    }
    return false;
}
