<?php
// public/admin/submissions_list.php
require_once __DIR__ . '/../../includes/admin_bootstrap.php';
$role = get_current_role();

// View permission
$can_view = (
  can_manage_finance($role) ||
  can_review_submissions($role) ||
  can_approve_submissions($role) ||
  in_array($role, ['Administrator','Super Administrator','Supervisor'], true)
);
if (!$can_view) { http_response_code(403); exit('Forbidden'); }

// Helpers
if (!function_exists('status_badge')) {
  function status_badge($s) {
    $s = strtolower((string)$s);
    $cls = $s==='approved' ? 'success' : ($s==='pending' ? 'warning' : ($s==='rejected' ? 'danger' : 'secondary'));
    return '<span class="badge text-bg-'.$cls.'">'.htmlspecialchars(ucfirst($s), ENT_QUOTES, 'UTF-8').'</span>';
  }
}
function bank_badge($validated) {
  return (int)$validated === 1
    ? '<span class="badge text-bg-success">TRX matched</span>'
    : '<span class="badge text-bg-secondary">TRX not matched</span>';
}
function table_exists_simple(mysqli $conn, $name) {
  $name = $conn->real_escape_string($name);
  $rs = $conn->query("SHOW TABLES LIKE '$name'");
  return $rs && $rs->num_rows > 0;
}

// Guards
if (!table_exists_simple($conn, 'submissions') || !table_exists_simple($conn, 'drivers')) {
  http_response_code(500);
  exit('Missing required tables.');
}

// Filters
$status = isset($_GET['status']) && in_array($_GET['status'], ['all','pending','rejected','approved'], true)
  ? $_GET['status'] : 'all';
$from   = isset($_GET['from']) && preg_match('/^\d{4}-\d{2}-\d{2}$/', $_GET['from']) ? $_GET['from'] : '';
$to     = isset($_GET['to'])   && preg_match('/^\d{4}-\d{2}-\d{2}$/', $_GET['to'])   ? $_GET['to']   : '';
$q      = trim($_GET['q'] ?? '');

// WHERE
$where = "1=1";
if ($status !== 'all') {
  $where .= " AND s.status='".$conn->real_escape_string($status)."'";
}
if ($from !== '') {
  $where .= " AND s.service_date >= '".$conn->real_escape_string($from)."'";
}
if ($to !== '') {
  $where .= " AND s.service_date <= '".$conn->real_escape_string($to)."'";
}
if ($q !== '') {
  $q_like = '%'.$conn->real_escape_string($q).'%';
  $where .= " AND (d.partner_id LIKE '$q_like' OR d.partner_uuid LIKE '$q_like' OR d.first_name LIKE '$q_like' OR d.last_name LIKE '$q_like')";
}

// EXPORT CSV
if (isset($_GET['export']) && $_GET['export'] === '1') {
  $exp_sql = "
    SELECT
      s.service_date       AS service_date,
      d.partner_id         AS driver_id,
      CONCAT(COALESCE(d.first_name,''), ' ', COALESCE(d.last_name,'')) AS driver_name,
      s.orders_count       AS orders,
      s.cash_deposited     AS cash_amount,
      s.bank_trx_no        AS bank_trx_no,
      s.trx_no             AS driver_trx_no,
      d.partner_uuid       AS partner_uuid,
      s.created_at         AS submitted_at,
      s.remarks            AS remarks,
      s.status             AS status
    FROM submissions s
    JOIN drivers d ON d.id = s.driver_id
    WHERE $where
    ORDER BY s.service_date DESC, s.id DESC
  ";
  $rs = $conn->query($exp_sql);

  // Output headers
  header('Content-Type: text/csv; charset=UTF-8');
  header('Content-Disposition: attachment; filename="submissions_export.csv"');
  // BOM for Excel
  echo "\xEF\xBB\xBF";

  $out = fopen('php://output', 'w');
  fputcsv(
    $out,
    ['date','driver id','driver name','partner uuid','orders','cash amount','bank trx no','driver trx no','submission timestamp','remarks','status']
  );

  if ($rs) {
    while ($r = $rs->fetch_assoc()) {
      fputcsv($out, [
        $r['service_date'],
        $r['driver_id'],
        $r['partner_uuid'],
        trim($r['driver_name']),
        (int)$r['orders'],
        number_format((float)$r['cash_amount'], 3, '.', ''), // OMR 3dp
        $r['bank_trx_no'],
        $r['driver_trx_no'],
        $r['submitted_at'],
        $r['remarks'],
        $r['status'],
      ]);
    }
  }
  fclose($out);
  exit;
}

// Pagination
$per_page = 200;
$page = isset($_GET['page']) && ctype_digit($_GET['page']) && (int)$_GET['page'] > 0 ? (int)$_GET['page'] : 1;
$offset = ($page - 1) * $per_page;

// Count
$count_sql = "
SELECT COUNT(*) AS cnt
FROM submissions s
JOIN drivers d ON d.id = s.driver_id
WHERE $where";
$count_res = $conn->query($count_sql);
$total = ($count_res && ($row = $count_res->fetch_assoc())) ? (int)$row['cnt'] : 0;
$pages = max(1, (int)ceil($total / $per_page));

// Data
$sql = "
SELECT
  s.id, s.driver_id, s.service_date, s.orders_count, s.cash_deposited,
  s.ss_orders, s.ss_deposit_1, s.ss_deposit_2,
  s.bank_trx_no, s.trx_no AS driver_trx_no,
  s.bank_validated, s.bank_validated_at, s.bank_amount, s.bank_date, s.bank_note,
  s.remarks, s.status, s.reviewed_by, s.reviewed_at, s.created_at,
  d.partner_id, d.partner_uuid, d.first_name, d.last_name
FROM submissions s
JOIN drivers d ON d.id = s.driver_id
WHERE $where
ORDER BY s.service_date DESC, s.id DESC
LIMIT $per_page OFFSET $offset";
$res = $conn->query($sql);
$rows = $res ? $res->fetch_all(MYSQLI_ASSOC) : [];

// Helper for query string without page
function qs_keep($add = []) {
  $base = $_GET;
  foreach ($add as $k=>$v) $base[$k] = $v;
  return http_build_query($base);
}

if (!function_exists('_sl_preview')) {
  function _sl_preview(?string $path): string {
    if (!$path) return '<div class="text-muted">No file</div>';
    $url = file_url($path);
    $ext = strtolower(pathinfo(parse_url($url, PHP_URL_PATH) ?? '', PATHINFO_EXTENSION));
    if (in_array($ext, ['jpg','jpeg','png','gif','webp'])) {
      return '<img src="'.e($url).'" class="img-fluid rounded border" alt="preview">';
    }
    if ($ext === 'pdf') {
      return '<div class="ratio ratio-4x3"><iframe src="'.e($url).'" class="rounded border" title="PDF"></iframe></div>';
    }
    return '<a class="btn btn-outline-secondary" href="'.e($url).'" target="_blank" rel="noopener">Open file</a>';
  }
}

?>
<!doctype html>
<html lang="en">
<head>
  <meta charset="utf-8">
  <title>Submissions list | Hisbuu</title>
  <meta name="viewport" content="width=device-width, initial-scale=1">
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
  <style>
    .nowrap { white-space: nowrap; }
    .smallmuted { font-size: .85rem; color: #6c757d; }
    .w-160 { width: 160px; }
    .w-200 { width: 200px; }
    .w-240 { width: 240px; }
    .pagination { margin: 0; }
  </style>
</head>
<body class="bg-light">
<?php include __DIR__ . '/../../includes/admin_nav.php'; ?>

<div class="container py-4" style="max-width: 1250px;">

  <div class="d-flex justify-content-between align-items-center mb-3">
    <h5 class="mb-0">Submissions list</h5>
    <div class="d-flex gap-2">
      <a href="<?php echo base_url('dashboard.php'); ?>" class="btn btn-outline-secondary btn-sm">Back</a>
      <!-- Export button preserves filters -->
      <a class="btn btn-primary btn-sm" href="?<?php echo qs_keep(['export'=>'1','page'=>1]); ?>">Export CSV</a>
    </div>
  </div>

  <div class="card mb-3">
    <div class="card-body">
      <form class="row g-2 align-items-end" method="get">
        <div class="col-auto">
          <label class="form-label d-block">Status</label>
          <select name="status" class="form-select form-select-sm" onchange="this.form.submit()">
            <option value="all"      <?php echo $status==='all'?'selected':''; ?>>All</option>
            <option value="pending"  <?php echo $status==='pending'?'selected':''; ?>>Pending</option>
            <option value="rejected" <?php echo $status==='rejected'?'selected':''; ?>>Rejected</option>
            <option value="approved" <?php echo $status==='approved'?'selected':''; ?>>Approved</option>
          </select>
        </div>
        <div class="col-auto">
          <label class="form-label">From</label>
          <input type="date" name="from" value="<?php echo e($from); ?>" class="form-control form-control-sm">
        </div>
        <div class="col-auto">
          <label class="form-label">To</label>
          <input type="date" name="to" value="<?php echo e($to); ?>" class="form-control form-control-sm">
        </div>
        <div class="col-auto">
          <label class="form-label">Search</label>
          <input type="text" name="q" value="<?php echo e($q); ?>" class="form-control form-control-sm" placeholder="Driver ID, UUID or name">
        </div>
        <div class="col-auto">
          <button class="btn btn-sm btn-primary">Filter</button>
        </div>
        <div class="col ms-auto text-end">
          <div class="small text-muted">
            Showing <?php echo count($rows); ?> of <?php echo $total; ?>, page <?php echo $page; ?> / <?php echo $pages; ?>
          </div>
        </div>
      </form>
    </div>
  </div>

  <div class="card">
    <div class="card-body">
      <div class="table-responsive">
        <table class="table table-sm table-striped align-middle mb-0">
          <thead>
            <tr>
              <th class="nowrap">Date</th>
              <th>Driver</th>
              <th>Partner UUID</th>
              <th class="text-end w-160">Orders</th>
              <th class="text-end w-200">Amount</th>
              <th style="min-width: 220px;">TRX No</th>
              <th class="nowrap w-160">Attachments</th>
              <th class="nowrap">Information</th>
              <th class="nowrap">Status</th>
            </tr>
          </thead>
          <tbody>
          <?php if (!$rows): ?>
            <tr><td colspan="9" class="text-muted">No records.</td></tr>
          <?php else: foreach ($rows as $r): ?>
            <?php
              $tip_html = 'Remarks: '.htmlspecialchars($r['remarks'] ?? '', ENT_QUOTES, 'UTF-8')
                        . '<br>Submitted: '.htmlspecialchars($r['created_at'] ?? '', ENT_QUOTES, 'UTF-8');
            ?>
            <tr>
              <td class="nowrap"><?php echo e($r['service_date']); ?></td>
              <td class="small">
                <div><strong><?php echo e(trim(($r['first_name']??'').' '.($r['last_name']??''))); ?></strong></div>
                <div class="smallmuted"><?php echo e($r['partner_id']); ?></div>
              </td>
              <td class="small"><?php echo e($r['partner_uuid']); ?></td>
              <td class="text-end"><?php echo (int)$r['orders_count']; ?></td>
              <td class="text-end"><?php echo number_format((float)$r['cash_deposited'], 3); ?></td>
              <td>
                <?php
                  $bankTrx = trim((string)($r['bank_trx_no'] ?? ''));
                  $driverTrx = trim((string)($r['driver_trx_no'] ?? ''));
                ?>
                <?php if ($bankTrx !== ''): ?>
                  <div><strong><?php echo e($bankTrx); ?></strong></div>
                  <div class="smallmuted"><?php echo bank_badge($r['bank_validated']); ?></div>
                  <?php if ((int)$r['bank_validated']===1): ?>
                    <div class="smallmuted">
                      <?php
                        $bits = [];
                        if (!is_null($r['bank_amount'])) $bits[] = 'OMR '.number_format((float)$r['bank_amount'],3);
                        if (!is_null($r['bank_date']))   $bits[] = e($r['bank_date']);
                        echo $bits ? implode(' &middot; ', $bits) : '';
                      ?>
                    </div>
                  <?php endif; ?>
                <?php elseif ($driverTrx !== ''): ?>
                  <div><strong><?php echo e($driverTrx); ?></strong></div>
                  <div class="smallmuted text-muted">Driver provided</div>
                <?php else: ?>
                  <span class="badge text-bg-light">No TRX</span>
                <?php endif; ?>
              </td>
              <?php $sid = (int)$r['id']; ?>
              <td class="nowrap">
                <button class="btn btn-outline-secondary btn-sm" data-bs-toggle="modal" data-bs-target="#mOrders_<?php echo $sid; ?>">Orders</button>
                <button class="btn btn-outline-secondary btn-sm" data-bs-toggle="modal" data-bs-target="#mDep1_<?php echo $sid; ?>">Dep 1</button>
                <button class="btn btn-outline-secondary btn-sm" data-bs-toggle="modal" data-bs-target="#mDep2_<?php echo $sid; ?>">Dep 2</button>
              </td>
              <td class="text-center">
                <span data-bs-toggle="tooltip"
                      data-bs-html="true"
                      title="<?php echo $tip_html; ?>"
                      style="cursor:pointer;">
                  <svg xmlns="http://www.w3.org/2000/svg" width="25" height="25"
                       fill="none" stroke="currentColor" stroke-width="2"
                       stroke-linecap="round" stroke-linejoin="round"
                       class="text-info">
                    <circle cx="12" cy="12" r="10"></circle>
                    <line x1="12" y1="16" x2="12" y2="12"></line>
                    <line x1="12" y1="8" x2="12" y2="8"></line>
                  </svg>
                </span>
              </td>
              <td><?php echo status_badge($r['status']); ?></td>
            </tr>
            <?php
  $ordersPath = $r['ss_orders'] ?? null;
  $dep1Path   = $r['ss_deposit_1'] ?? null;
  $dep2Path   = $r['ss_deposit_2'] ?? null;
?>

<!-- Orders modal -->
<div class="modal fade" id="mOrders_<?php echo $sid; ?>" tabindex="-1" aria-hidden="true">
  <div class="modal-dialog modal-lg modal-dialog-centered"><div class="modal-content">
    <div class="modal-header">
      <h6 class="modal-title">Order screenshot · <?php echo e($r['service_date']); ?></h6>
      <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
    </div>
    <div class="modal-body">
      <?php echo _sl_preview($ordersPath); ?>
    </div>
  </div></div>
</div>

<!-- Deposit 1 modal -->
<div class="modal fade" id="mDep1_<?php echo $sid; ?>" tabindex="-1" aria-hidden="true">
  <div class="modal-dialog modal-lg modal-dialog-centered"><div class="modal-content">
    <div class="modal-header">
      <h6 class="modal-title">Deposit slip 1 · <?php echo e($r['service_date']); ?></h6>
      <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
    </div>
    <div class="modal-body">
      <?php echo _sl_preview($dep1Path); ?>
    </div>
  </div></div>
</div>

<!-- Deposit 2 modal -->
<div class="modal fade" id="mDep2_<?php echo $sid; ?>" tabindex="-1" aria-hidden="true">
  <div class="modal-dialog modal-lg modal-dialog-centered"><div class="modal-content">
    <div class="modal-header">
      <h6 class="modal-title">Deposit slip 2 · <?php echo e($r['service_date']); ?></h6>
      <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
    </div>
    <div class="modal-body">
      <?php echo _sl_preview($dep2Path); ?>
    </div>
  </div></div>
</div>

          <?php endforeach; endif; ?>
          </tbody>
        </table>
      </div>

      <!-- Pagination -->
      <?php if ($pages > 1): ?>
      <div class="d-flex justify-content-between align-items-center mt-3">
        <div class="small text-muted">Total <?php echo $total; ?> records</div>
        <nav aria-label="Page nav">
          <ul class="pagination pagination-sm mb-0">
            <?php
              $prev = max(1, $page - 1);
              $next = min($pages, $page + 1);
            ?>
            <li class="page-item <?php echo $page<=1?'disabled':''; ?>">
              <a class="page-link" href="?<?php echo qs_keep(['page'=>$prev]); ?>">Prev</a>
            </li>
            <?php for ($p=max(1,$page-2); $p<=min($pages,$page+2); $p++): ?>
              <li class="page-item <?php echo $p===$page?'active':''; ?>">
                <a class="page-link" href="?<?php echo qs_keep(['page'=>$p]); ?>"><?php echo $p; ?></a>
              </li>
            <?php endfor; ?>
            <li class="page-item <?php echo $page>=$pages?'disabled':''; ?>">
              <a class="page-link" href="?<?php echo qs_keep(['page'=>$next]); ?>">Next</a>
            </li>
          </ul>
        </nav>
      </div>
      <?php endif; ?>

    </div>
  </div>
</div>
<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"></script>
<script>
document.addEventListener('DOMContentLoaded', function () {
  const tooltipTriggerList = [].slice.call(document.querySelectorAll('[data-bs-toggle="tooltip"]'));
  tooltipTriggerList.forEach(function (el) { new bootstrap.Tooltip(el); });
});
</script>
</body>
</html>







