﻿<?php
require_once __DIR__ . '/../includes/admin_bootstrap.php';

$role = get_current_role();

function h($s) { return htmlspecialchars((string)$s, ENT_QUOTES, 'UTF-8'); }

function table_exists_simple(mysqli $conn, string $name): bool {
  $name = $conn->real_escape_string($name);
  $rs = $conn->query("SHOW TABLES LIKE '$name'");
  return $rs && $rs->num_rows > 0;
}

// Small helpers for counts, safe even if a table is missing
function table_count(mysqli $conn, string $table): ?int {
  $res = $conn->query("SHOW TABLES LIKE '{$conn->real_escape_string($table)}'");
  if (!$res || $res->num_rows === 0) return null;
  $r = $conn->query("SELECT COUNT(*) AS c FROM {$table}");
  if (!$r) return null;
  $row = $r->fetch_assoc();
  return (int)($row['c'] ?? 0);
}

function pending_count(mysqli $conn, string $table, string $col = 'status', string $val = 'pending'): ?int {
  $res = $conn->query("SHOW TABLES LIKE '{$conn->real_escape_string($table)}'");
  if (!$res || $res->num_rows === 0) return null;
  $valEsc = $conn->real_escape_string($val);
  $q = $conn->query("SELECT COUNT(*) AS c FROM {$table} WHERE {$col}='{$valEsc}'");
  if (!$q) return null;
  $row = $q->fetch_assoc();
  return (int)($row['c'] ?? 0);
}

function display_count(?int $val): string {
  return $val === null ? '-' : number_format($val);
}

function display_cash(?float $val): string {
  return $val === null ? '-' : number_format($val, 3);
}

$has_drivers = table_exists_simple($conn, 'drivers');
$has_submissions = table_exists_simple($conn, 'submissions');
$has_partner = table_exists_simple($conn, 'partner_daily');

// Pull quick stats if tables exist
$drivers_total = $has_drivers ? table_count($conn, 'drivers') : null;
$subs_total = $has_submissions ? table_count($conn, 'submissions') : null;
$subs_pending = $has_submissions ? pending_count($conn, 'submissions', 'status', 'pending') : null;
$partner_rows = $has_partner ? table_count($conn, 'partner_daily') : null;
$partner_pending = $has_partner ? pending_count($conn, 'partner_daily', 'status', 'pending') : null;

$cash_30d = null;
$orders_30d = null;
if ($has_submissions) {
  $rs = $conn->query(
    "SELECT SUM(orders_count) AS orders, SUM(cash_deposited) AS cash
     FROM submissions
     WHERE service_date >= DATE_SUB(CURDATE(), INTERVAL 30 DAY)"
  );
  if ($rs) {
    $row = $rs->fetch_assoc();
    $orders_30d = (int)($row['orders'] ?? 0);
    $cash_30d = (float)($row['cash'] ?? 0);
  }
}

// Trend data (last 6 months)
$trend_map = [];
if ($has_submissions) {
  $trend_sql =
    "SELECT DATE_FORMAT(service_date, '%Y-%m') AS ym,
            SUM(orders_count) AS orders,
            SUM(cash_deposited) AS cash
     FROM submissions
     WHERE service_date >= DATE_SUB(CURDATE(), INTERVAL 5 MONTH)
     GROUP BY ym
     ORDER BY ym";
  $trend_res = $conn->query($trend_sql);
  if ($trend_res) {
    while ($row = $trend_res->fetch_assoc()) {
      $ym = (string)($row['ym'] ?? '');
      if ($ym === '') continue;
      $trend_map[$ym] = [
        'orders' => (int)($row['orders'] ?? 0),
        'cash' => (float)($row['cash'] ?? 0),
      ];
    }
  }
}

$trend_labels = [];
$trend_orders = [];
$trend_cash = [];
$cursor = new DateTime('first day of this month');
$cursor->modify('-5 months');
for ($i = 0; $i < 6; $i++) {
  $ym = $cursor->format('Y-m');
  $trend_labels[] = $cursor->format('M Y');
  $trend_orders[] = (int)($trend_map[$ym]['orders'] ?? 0);
  $trend_cash[] = (float)($trend_map[$ym]['cash'] ?? 0);
  $cursor->modify('+1 month');
}

// Status breakdown (last 30 days)
$status_map = [
  'pending' => 0,
  'approved' => 0,
  'rejected' => 0,
  'other' => 0,
];
if ($has_submissions) {
  $status_sql =
    "SELECT status, COUNT(*) AS cnt
     FROM submissions
     WHERE service_date >= DATE_SUB(CURDATE(), INTERVAL 30 DAY)
     GROUP BY status";
  $status_res = $conn->query($status_sql);
  if ($status_res) {
    while ($row = $status_res->fetch_assoc()) {
      $key = strtolower(trim((string)($row['status'] ?? '')));
      if ($key === '' || !array_key_exists($key, $status_map)) {
        $key = 'other';
      }
      $status_map[$key] += (int)($row['cnt'] ?? 0);
    }
  }
}

$status_labels = ['Pending', 'Approved', 'Rejected'];
$status_counts = [
  (int)$status_map['pending'],
  (int)$status_map['approved'],
  (int)$status_map['rejected'],
];
if ($status_map['other'] > 0) {
  $status_labels[] = 'Other';
  $status_counts[] = (int)$status_map['other'];
}

// Top drivers (last 30 days)
$top_driver_labels = [];
$top_driver_orders = [];
$top_driver_rows = [];
if ($has_submissions && $has_drivers) {
  $top_sql =
    "SELECT CONCAT(COALESCE(d.first_name,''), ' ', COALESCE(d.last_name,'')) AS driver_name,
            SUM(s.orders_count) AS orders
     FROM submissions s
     JOIN drivers d ON d.id = s.driver_id
     WHERE s.service_date >= DATE_SUB(CURDATE(), INTERVAL 30 DAY)
     GROUP BY s.driver_id
     ORDER BY orders DESC
     LIMIT 5";
  $top_res = $conn->query($top_sql);
  if ($top_res) {
    while ($row = $top_res->fetch_assoc()) {
      $name = trim((string)($row['driver_name'] ?? ''));
      if ($name === '') $name = 'Unknown';
      $orders = (int)($row['orders'] ?? 0);
      $top_driver_labels[] = $name;
      $top_driver_orders[] = $orders;
      $top_driver_rows[] = ['name' => $name, 'orders' => $orders];
    }
  }
}

// Permissions
$canDrivers = can_manage_drivers($role) || can_review_submissions($role);
$canApprovals = can_review_submissions($role) || can_approve_submissions($role);
$canPartner = can_view_partner_daily($role) || can_manage_partner_daily($role);
$canLookups = can_view_lookups($role);
$canFinance = can_manage_finance($role);
$canUsers = can_manage_users($role);
$canSettings = can_edit_settings($role);

$PAGE_TITLE = 'Admin dashboard';
require_once __DIR__ . '/../includes/admin_header.php';
?>

<style>
.dashboard-kpi {
  border: 1px solid #f0f0f0;
}
.dashboard-kpi .kpi-icon {
  width: 42px;
  height: 42px;
  border-radius: 12px;
  background: rgba(194, 15, 47, 0.12);
  color: var(--hisbuu-red);
  display: flex;
  align-items: center;
  justify-content: center;
  font-size: 18px;
}
.dashboard-kpi .kpi-label {
  font-size: 0.85rem;
  color: #6c757d;
}
.dashboard-kpi .kpi-value {
  font-size: 1.6rem;
  font-weight: 700;
}
.chart-card {
  min-height: 320px;
}
.chart-canvas {
  width: 100%;
  height: 240px;
}
.queue-item {
  display: flex;
  align-items: center;
  justify-content: space-between;
  padding: 6px 0;
  border-bottom: 1px dashed #e9ecef;
}
.queue-item:last-child {
  border-bottom: 0;
}
.quick-link {
  border: 1px solid #f0f0f0;
  background: #fff;
}
.quick-link:hover {
  border-color: #e2e2e2;
}
</style>

<div class="page-head">
  <div>
    <h5>Admin dashboard</h5>
    <div class="page-sub">Signed in as <?php echo h($role); ?></div>
  </div>
  <div class="text-muted small">Updated <?php echo date('d M Y, H:i'); ?></div>
</div>

<div class="row g-3">
  <div class="col-md-3">
    <div class="card shadow-sm dashboard-kpi h-100">
      <div class="card-body d-flex gap-3 align-items-start">
        <div class="kpi-icon"><i class="bi bi-people"></i></div>
        <div>
          <div class="kpi-label">Drivers</div>
          <div class="kpi-value"><?php echo display_count($drivers_total); ?></div>
          <?php if ($canDrivers): ?>
            <a href="<?php echo base_url('admin/driver_list.php'); ?>" class="btn btn-outline-secondary btn-sm mt-2">Manage</a>
          <?php endif; ?>
        </div>
      </div>
    </div>
  </div>
  <div class="col-md-3">
    <div class="card shadow-sm dashboard-kpi h-100">
      <div class="card-body d-flex gap-3 align-items-start">
        <div class="kpi-icon"><i class="bi bi-hourglass-split"></i></div>
        <div>
          <div class="kpi-label">Pending approvals</div>
          <div class="kpi-value"><?php echo display_count($subs_pending); ?></div>
          <?php if ($canApprovals): ?>
            <a href="<?php echo base_url('admin/pending_approvals.php'); ?>" class="btn btn-outline-secondary btn-sm mt-2">Review</a>
          <?php endif; ?>
        </div>
      </div>
    </div>
  </div>
  <div class="col-md-3">
    <div class="card shadow-sm dashboard-kpi h-100">
      <div class="card-body d-flex gap-3 align-items-start">
        <div class="kpi-icon"><i class="bi bi-inbox"></i></div>
        <div>
          <div class="kpi-label">Submissions total</div>
          <div class="kpi-value"><?php echo display_count($subs_total); ?></div>
          <?php if ($canApprovals): ?>
            <a href="<?php echo base_url('admin/submissions_list.php'); ?>" class="btn btn-outline-secondary btn-sm mt-2">View all</a>
          <?php endif; ?>
        </div>
      </div>
    </div>
  </div>
  <div class="col-md-3">
    <div class="card shadow-sm dashboard-kpi h-100">
      <div class="card-body d-flex gap-3 align-items-start">
        <div class="kpi-icon"><i class="bi bi-cash-stack"></i></div>
        <div>
          <div class="kpi-label">Cash deposited (30d)</div>
          <div class="kpi-value"><?php echo display_cash($cash_30d); ?></div>
          <div class="text-muted small">Orders <?php echo display_count($orders_30d); ?></div>
        </div>
      </div>
    </div>
  </div>
</div>

<div class="row g-3 mt-1">
  <div class="col-lg-8">
    <div class="card shadow-sm chart-card h-100">
      <div class="card-body">
        <div class="d-flex justify-content-between align-items-center mb-2">
          <div>
            <div class="fw-semibold">Submissions trend</div>
            <div class="text-muted small">Last 6 months</div>
          </div>
          <span class="badge text-bg-light">Orders and cash</span>
        </div>
        <div class="chart-wrap">
          <canvas id="trendChart" class="chart-canvas"></canvas>
        </div>
      </div>
    </div>
  </div>
  <div class="col-lg-4">
    <div class="card shadow-sm chart-card h-100">
      <div class="card-body">
        <div class="fw-semibold">Status mix</div>
        <div class="text-muted small mb-3">Last 30 days</div>
        <div class="chart-wrap">
          <canvas id="statusChart" class="chart-canvas"></canvas>
        </div>
        <div class="mt-3 small">
          <div class="queue-item"><span>Pending</span><span><?php echo display_count($status_map['pending']); ?></span></div>
          <div class="queue-item"><span>Approved</span><span><?php echo display_count($status_map['approved']); ?></span></div>
          <div class="queue-item"><span>Rejected</span><span><?php echo display_count($status_map['rejected']); ?></span></div>
          <?php if ($status_map['other'] > 0): ?>
            <div class="queue-item"><span>Other</span><span><?php echo display_count($status_map['other']); ?></span></div>
          <?php endif; ?>
        </div>
      </div>
    </div>
  </div>
</div>

<div class="row g-3 mt-1">
  <div class="col-lg-7">
    <div class="card shadow-sm chart-card h-100">
      <div class="card-body">
        <div class="d-flex justify-content-between align-items-center mb-2">
          <div>
            <div class="fw-semibold">Top drivers</div>
            <div class="text-muted small">Orders in last 30 days</div>
          </div>
          <span class="badge text-bg-light"><?php echo display_count(array_sum($top_driver_orders)); ?> orders</span>
        </div>
        <div class="chart-wrap">
          <canvas id="driversChart" class="chart-canvas"></canvas>
        </div>
        <?php if (!empty($top_driver_rows)): ?>
          <div class="mt-3">
            <?php foreach ($top_driver_rows as $r): ?>
              <div class="queue-item">
                <span><?php echo h($r['name']); ?></span>
                <span><?php echo display_count($r['orders']); ?></span>
              </div>
            <?php endforeach; ?>
          </div>
        <?php else: ?>
          <div class="text-muted small mt-3">No recent submissions yet.</div>
        <?php endif; ?>
      </div>
    </div>
  </div>
  <div class="col-lg-5">
    <div class="card shadow-sm h-100">
      <div class="card-body">
        <div class="fw-semibold mb-2">Work queue</div>
        <div class="queue-item">
          <span>Pending submissions</span>
          <span class="badge text-bg-warning"><?php echo display_count($subs_pending); ?></span>
        </div>
        <div class="queue-item">
          <span>Partner rows pending</span>
          <span class="badge text-bg-info"><?php echo display_count($partner_pending); ?></span>
        </div>
        <div class="queue-item">
          <span>Partner rows total</span>
          <span><?php echo display_count($partner_rows); ?></span>
        </div>
        <div class="mt-3 d-flex flex-wrap gap-2">
          <?php if ($canApprovals): ?>
            <a href="<?php echo base_url('admin/pending_approvals.php'); ?>" class="btn btn-outline-secondary btn-sm">Review submissions</a>
          <?php endif; ?>
          <?php if ($canPartner): ?>
            <a href="<?php echo base_url('admin/data_entry.php'); ?>" class="btn btn-outline-secondary btn-sm">Partner data</a>
          <?php endif; ?>
          <?php if ($canFinance): ?>
            <a href="<?php echo base_url('admin/bank_upload.php'); ?>" class="btn btn-outline-secondary btn-sm">Bank matcher</a>
          <?php endif; ?>
        </div>
      </div>
    </div>
  </div>
</div>

<div class="card shadow-sm mt-4">
  <div class="card-body">
    <div class="fw-semibold mb-3">Quick links</div>
    <div class="row g-2">
      <?php if ($canDrivers): ?>
        <div class="col-md-4"><a class="btn btn-light quick-link w-100 text-start" href="<?php echo base_url('admin/driver_list.php'); ?>"><i class="bi bi-people me-2 text-muted"></i>Drivers</a></div>
        <div class="col-md-4"><a class="btn btn-light quick-link w-100 text-start" href="<?php echo base_url('admin/driver_create.php'); ?>"><i class="bi bi-person-plus me-2 text-muted"></i>Create driver</a></div>
      <?php endif; ?>

      <?php if ($canApprovals): ?>
        <div class="col-md-4"><a class="btn btn-light quick-link w-100 text-start" href="<?php echo base_url('admin/pending_approvals.php'); ?>"><i class="bi bi-hourglass-split me-2 text-muted"></i>Pending approvals</a></div>
        <div class="col-md-4"><a class="btn btn-light quick-link w-100 text-start" href="<?php echo base_url('admin/submissions_list.php'); ?>"><i class="bi bi-inbox me-2 text-muted"></i>Submissions list</a></div>
        <div class="col-md-4"><a class="btn btn-light quick-link w-100 text-start" href="<?php echo base_url('admin/submissions_summary.php'); ?>"><i class="bi bi-calendar3 me-2 text-muted"></i>Monthly summary</a></div>
      <?php endif; ?>

      <?php if ($canPartner): ?>
        <div class="col-md-4"><a class="btn btn-light quick-link w-100 text-start" href="<?php echo base_url('admin/data_entry.php'); ?>"><i class="bi bi-diagram-3 me-2 text-muted"></i>Partner data entry</a></div>
        <div class="col-md-4"><a class="btn btn-light quick-link w-100 text-start" href="<?php echo base_url('admin/partner_upload.php'); ?>"><i class="bi bi-upload me-2 text-muted"></i>Bulk upload CSV/XLSX</a></div>
        <div class="col-md-4"><a class="btn btn-light quick-link w-100 text-start" href="<?php echo base_url('admin/partner_template.php'); ?>"><i class="bi bi-download me-2 text-muted"></i>Download CSV template</a></div>
      <?php endif; ?>

      <?php if ($canLookups): ?>
        <div class="col-md-4"><a class="btn btn-light quick-link w-100 text-start" href="<?php echo base_url('admin/projects_manage.php'); ?>"><i class="bi bi-diagram-2 me-2 text-muted"></i>Projects</a></div>
        <div class="col-md-4"><a class="btn btn-light quick-link w-100 text-start" href="<?php echo base_url('admin/types_manage.php'); ?>"><i class="bi bi-list-check me-2 text-muted"></i>Types</a></div>
        <div class="col-md-4"><a class="btn btn-light quick-link w-100 text-start" href="<?php echo base_url('admin/cities_manage.php'); ?>"><i class="bi bi-geo-alt me-2 text-muted"></i>Cities</a></div>
        <div class="col-md-4"><a class="btn btn-light quick-link w-100 text-start" href="<?php echo base_url('admin/zones_manage.php'); ?>"><i class="bi bi-grid-1x2 me-2 text-muted"></i>Zones</a></div>
        <div class="col-md-4"><a class="btn btn-light quick-link w-100 text-start" href="<?php echo base_url('admin/nationalities_manage.php'); ?>"><i class="bi bi-flag me-2 text-muted"></i>Nationalities</a></div>
        <div class="col-md-4"><a class="btn btn-light quick-link w-100 text-start" href="<?php echo base_url('admin/salary_schemes_manage.php'); ?>"><i class="bi bi-wallet2 me-2 text-muted"></i>Salary schemes</a></div>
      <?php endif; ?>

      <?php if ($canFinance): ?>
        <div class="col-md-4"><a class="btn btn-light quick-link w-100 text-start" href="<?php echo base_url('admin/finance_reports.php'); ?>"><i class="bi bi-graph-up-arrow me-2 text-muted"></i>Finance reports</a></div>
      <?php endif; ?>

      <?php if ($canUsers): ?>
        <div class="col-md-4"><a class="btn btn-light quick-link w-100 text-start" href="<?php echo base_url('admin/users_manage.php'); ?>"><i class="bi bi-person-gear me-2 text-muted"></i>Users</a></div>
      <?php endif; ?>

      <?php if ($canSettings): ?>
        <div class="col-md-4"><a class="btn btn-light quick-link w-100 text-start" href="<?php echo base_url('admin/settings.php'); ?>"><i class="bi bi-gear me-2 text-muted"></i>Settings</a></div>
      <?php endif; ?>
    </div>
  </div>
</div>

<div class="page-sub small mt-3">Admin area only. Drivers should use driver/login.php.</div>

<script src="https://cdn.jsdelivr.net/npm/chart.js@4.4.1/dist/chart.umd.min.js"></script>
<script>
(function () {
  if (!window.Chart) return;

  const trendLabels = <?php echo json_encode($trend_labels); ?>;
  const trendOrders = <?php echo json_encode($trend_orders); ?>;
  const trendCash = <?php echo json_encode($trend_cash); ?>;
  const statusLabels = <?php echo json_encode($status_labels); ?>;
  const statusCounts = <?php echo json_encode($status_counts); ?>;
  const driverLabels = <?php echo json_encode($top_driver_labels); ?>;
  const driverOrders = <?php echo json_encode($top_driver_orders); ?>;

  const trendCtx = document.getElementById('trendChart');
  if (trendCtx) {
    new Chart(trendCtx, {
      data: {
        labels: trendLabels,
        datasets: [
          {
            type: 'bar',
            label: 'Orders',
            data: trendOrders,
            backgroundColor: 'rgba(194, 15, 47, 0.2)',
            borderColor: '#C20F2F',
            borderWidth: 1,
            yAxisID: 'y'
          },
          {
            type: 'line',
            label: 'Cash (OMR)',
            data: trendCash,
            borderColor: '#0d6efd',
            backgroundColor: 'rgba(13, 110, 253, 0.15)',
            tension: 0.3,
            fill: true,
            yAxisID: 'y1'
          }
        ]
      },
      options: {
        maintainAspectRatio: false,
        scales: {
          y: {
            beginAtZero: true,
            grid: { color: 'rgba(0,0,0,0.05)' }
          },
          y1: {
            beginAtZero: true,
            position: 'right',
            grid: { drawOnChartArea: false }
          }
        }
      }
    });
  }

  const statusCtx = document.getElementById('statusChart');
  if (statusCtx) {
    new Chart(statusCtx, {
      type: 'doughnut',
      data: {
        labels: statusLabels,
        datasets: [{
          data: statusCounts,
          backgroundColor: ['#ffc107', '#198754', '#dc3545', '#6c757d']
        }]
      },
      options: {
        maintainAspectRatio: false,
        plugins: {
          legend: { position: 'bottom' }
        }
      }
    });
  }

  const driversCtx = document.getElementById('driversChart');
  if (driversCtx) {
    new Chart(driversCtx, {
      type: 'bar',
      data: {
        labels: driverLabels,
        datasets: [{
          label: 'Orders',
          data: driverOrders,
          backgroundColor: 'rgba(13, 110, 253, 0.2)',
          borderColor: '#0d6efd',
          borderWidth: 1
        }]
      },
      options: {
        maintainAspectRatio: false,
        indexAxis: 'y',
        scales: {
          x: { beginAtZero: true }
        },
        plugins: {
          legend: { display: false }
        }
      }
    });
  }
})();
</script>

<?php require_once __DIR__ . '/../includes/admin_footer.php'; ?>
