<?php
require_once __DIR__ . '/../../includes/driver_bootstrap.php';

$driver_id = get_current_driver_id();

// Filters
$currentMonth = date('Y-m');
$month = (isset($_GET['month']) && preg_match('/^\d{4}-\d{2}$/', $_GET['month']))
    ? $_GET['month']
    : $currentMonth;

$start = $month . '-01';
$end   = date('Y-m-t', strtotime($start));

// Combined partner_daily + submissions
$sql = "
  (SELECT
      pd.service_date,
      pd.orders_count AS partner_orders,
      pd.amount_total AS partner_amount,
      s.id,
      s.orders_count,
      s.cash_deposited,
      s.ss_orders,
      s.ss_deposit_1,
      s.ss_deposit_2,
      s.trx_no,
      s.remarks,
      s.status,
      s.created_at,
      s.is_trx_validated
   FROM partner_daily pd
   LEFT JOIN submissions s
     ON pd.service_date = s.service_date
     AND pd.driver_id = s.driver_id
   WHERE pd.driver_id = ? AND pd.service_date BETWEEN ? AND ?)

  UNION

  (SELECT
      s.service_date,
      NULL AS partner_orders,
      NULL AS partner_amount,
      s.id,
      s.orders_count,
      s.cash_deposited,
      s.ss_orders,
      s.ss_deposit_1,
      s.ss_deposit_2,
      s.trx_no,
      s.remarks,
      s.status,
      s.created_at,
      s.is_trx_validated
   FROM submissions s
   LEFT JOIN partner_daily pd
     ON s.service_date = pd.service_date
     AND s.driver_id = pd.driver_id
   WHERE s.driver_id = ? AND s.service_date BETWEEN ? AND ? AND pd.id IS NULL)

   ORDER BY service_date DESC
";

$stmt = $conn->prepare($sql);
$stmt->bind_param('ississ', $driver_id, $start, $end, $driver_id, $start, $end);
$stmt->execute();
$rows = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);

// Summary
$summary = ['orders'=>0, 'collection'=>0.0, 'deposit'=>0.0, 'balance'=>0.0];

// Total Orders - sum from approved submissions only
$sqlOrders = "
  SELECT COALESCE(SUM(orders_count),0)
  FROM submissions
  WHERE driver_id=? AND LOWER(status)='approved'
  AND service_date BETWEEN ? AND ?
";

// Total Collection - sum from partner_daily (partner data)
$sqlCollection = "
  SELECT COALESCE(SUM(amount_total),0)
  FROM partner_daily
  WHERE driver_id=? AND service_date BETWEEN ? AND ?
";

// Total Deposit - sum from approved submissions only
$sqlDeposit = "
  SELECT COALESCE(SUM(cash_deposited),0)
  FROM submissions
  WHERE driver_id=? AND LOWER(status)='approved'
  AND service_date BETWEEN ? AND ?
";

$stmt = $conn->prepare($sqlOrders);
$stmt->bind_param('iss', $driver_id, $start, $end);
$stmt->execute();
$summary['orders'] = (int)($stmt->get_result()->fetch_row()[0] ?? 0);
$stmt->close();

$stmt = $conn->prepare($sqlCollection);
$stmt->bind_param('iss', $driver_id, $start, $end);
$stmt->execute();
$summary['collection'] = (float)($stmt->get_result()->fetch_row()[0] ?? 0);
$stmt->close();

$stmt = $conn->prepare($sqlDeposit);
$stmt->bind_param('iss', $driver_id, $start, $end);
$stmt->execute();
$summary['deposit'] = (float)($stmt->get_result()->fetch_row()[0] ?? 0);
$stmt->close();

// Balance = Total Collection (Partner Data) - Total Deposited (Approved)
$summary['balance'] = $summary['collection'] - $summary['deposit'];

function n3($v) { return number_format((float)$v, 3, '.', ''); }

function badge($s) {
  $s = strtolower((string)$s);
  return match($s) {
    'yet to submit' => '<span class="badge text-bg-secondary">Yet to Submit</span>',
    'approved'      => '<span class="badge text-bg-success">Approved</span>',
    'rejected'      => '<span class="badge text-bg-danger">Rejected</span>',
    default         => '<span class="badge text-bg-warning text-dark">Pending</span>'
  };
}

function submission_attachments(array $row) {
  $out = [];
  $map = [
    'ss_orders'    => 'Orders slip',
    'ss_deposit_1' => 'Deposit slip 1',
    'ss_deposit_2' => 'Deposit slip 2'
  ];
  foreach ($map as $key => $label) {
    if (!empty($row[$key])) {
      $out[] = [
        'label' => $label,
        'url'   => file_url($row[$key])
      ];
    }
  }
  return $out;
}

$title = 'Submission history';
require_once __DIR__ . '/../../includes/driver_ui_start.php';
?>

<style>
/* Put modal above sticky top bar and bottom nav */
.modal {
  z-index: 9999 !important;
}

.modal-backdrop {
  z-index: 9998 !important;
}

/* Constrain modal height and make body scrollable */
.modal-content {
  max-height: calc(100vh - 130px);
  border-radius: 12px;
  display: flex;
  flex-direction: column;
  overflow: hidden;
}

.modal-body {
  overflow-y: auto;
  -webkit-overflow-scrolling: touch;
  padding-bottom: 20px;
}

/* Responsive image */
.file-preview-img {
  max-width: 100%;
  height: auto;
  max-height: 60vh;
  object-fit: contain;
}

/* Responsive iframe */
.file-preview-frame {
  width: 100% !important;
  height: 60vh !important;
  border: 0;
  border-radius: 6px;
}

/* Slightly reduce side margins on very small screens */
@media (max-width: 576px) {
  .modal-dialog {
    margin: 0.5rem auto;
  }
}
</style>

<div class="d-flex justify-content-between align-items-center mb-3">
  <div>
    <h5 class="mb-0">Submission History</h5>
    <div class="text-muted small">Track your daily orders, deposits, and approvals.</div>
  </div>
  <a class="btn btn-outline-secondary btn-sm" href="<?php echo base_url('driver/dashboard.php'); ?>">Back</a>
</div>

<!-- Filter -->
<div class="card mb-3">
  <div class="card-body">
    <form class="row g-3 align-items-end" method="get">
      <div class="col-md-3 col-sm-6">
        <label class="form-label">Filter by Month</label>
        <input type="month" name="month" class="form-control" value="<?php echo e($month); ?>">
      </div>
      <div class="col-md-3 col-sm-6">
        <button class="btn btn-outline-secondary" type="submit">Filter</button>
        <a href="history.php" class="btn btn-link btn-sm">Clear</a>
      </div>
    </form>
  </div>
</div>

<!-- Summary -->
<div class="row row-cols-1 row-cols-sm-2 row-cols-lg-4 g-3 mb-4">
  <div class="col">
    <div class="card shadow-sm border-0 h-100">
      <div class="card-body">
        <div class="text-muted small text-uppercase">Total Orders</div>
        <div class="fs-4 fw-semibold"><?php echo number_format($summary['orders']); ?></div>
      </div>
    </div>
  </div>

  <div class="col">
    <div class="card shadow-sm border-0 h-100">
      <div class="card-body">
        <div class="text-muted small text-uppercase">Total Cash</div>
        <div class="fs-4 fw-semibold">OMR <?php echo n3($summary['collection']); ?></div>
      </div>
    </div>
  </div>

  <div class="col">
    <div class="card shadow-sm border-0 h-100">
      <div class="card-body">
        <div class="text-muted small text-uppercase">Total Deposit</div>
        <div class="fs-4 fw-semibold">OMR <?php echo n3($summary['deposit']); ?></div>
      </div>
    </div>
  </div>

  <div class="col">
    <div class="card shadow-sm border-0 h-100">
      <div class="card-body">
        <div class="text-muted small text-uppercase">Balance</div>
        <?php
          $cls = $summary['balance'] > 0 ? 'text-danger'
               : ($summary['balance'] < 0 ? 'text-success' : '');
        ?>
        <div class="fs-4 fw-semibold <?php echo $cls; ?>">
          OMR <?php echo n3($summary['balance']); ?>
        </div>
      </div>
    </div>
  </div>
</div>

<!-- Status Filter -->
<div class="d-flex flex-wrap gap-2 mb-3" id="status-filter">
  <button class="btn btn-sm btn-dark active" data-status="all">All</button>
  <button class="btn btn-sm btn-outline-secondary" data-status="yet-to-submit">Yet to Submit</button>
  <button class="btn btn-sm btn-outline-secondary" data-status="pending">Pending</button>
  <button class="btn btn-sm btn-outline-secondary" data-status="approved">Approved</button>
  <button class="btn btn-sm btn-outline-secondary" data-status="rejected">Rejected</button>
</div>

<?php if (!$rows): ?>
  <div class="alert alert-light text-center">No history found.</div>
<?php else: ?>
  <div class="accordion" id="historyAccordion">
    <?php foreach ($rows as $r): ?>
      <?php
        $sid  = (int)$r['id'];
        $stat = $sid ? strtolower($r['status'] ?? 'pending') : 'yet to submit';
        $can_resubmit = in_array($stat, ['pending', 'rejected'], true);
        $attachments = submission_attachments($r);
        $attachmentsPayload = htmlspecialchars(json_encode($attachments), ENT_QUOTES);
        $accordionId = $sid ? 'item-' . $sid : 'item-' . str_replace('-', '', $r['service_date']);
        $status_slug = str_replace(' ', '-', $stat);
      ?>
      <div class="card mb-2 history-item" data-status="<?php echo $status_slug; ?>">
        <div class="card-header p-0">
          <div class="d-flex align-items-center p-3">
            <div class="flex-grow-1">
              <div class="row">
                <div class="col-6 col-md-3">
                  <div class="small text-muted">Date</div>
                  <div class="fw-semibold"><?php echo e($r['service_date']); ?></div>
                </div>
                <div class="col-6 col-md-2">
                  <div class="small text-muted">Status</div>
                  <div><?php echo badge($stat); ?></div>
                </div>
                <div class="col-6 col-md-2 mt-2 mt-md-0">
                  <div class="small text-muted">Orders</div>
                  <div class="fw-semibold">
                    <?php echo (int)($r['orders_count'] ?? $r['partner_orders']); ?>
                  </div>
                </div>
                <div class="col-6 col-md-3 mt-2 mt-md-0">
                  <div class="small text-muted">Cash</div>
                  <div class="fw-semibold">
                    <?php echo n3($r['cash_deposited'] ?? $r['partner_amount']); ?>
                  </div>
                </div>
              </div>
            </div>

            <button
              class="btn btn-sm btn-outline-secondary ms-2"
              type="button"
              data-bs-toggle="collapse"
              data-bs-target="#<?php echo $accordionId; ?>"
              aria-expanded="false"
              aria-controls="<?php echo $accordionId; ?>">
              Details
            </button>
          </div>
        </div>

        <div id="<?php echo $accordionId; ?>" class="collapse" data-bs-parent="#historyAccordion">
          <div class="card-body border-top">

            <?php if ($stat !== 'yet to submit'): ?>
              <div class="row">
                <div class="col-md-6">
                  <div class="small text-muted">TRX #</div>
                  <p>
                    <?php if (!empty($r['trx_no'])): ?>
                      <span class="d-flex align-items-center gap-1">
                        <span><?php echo e($r['trx_no']); ?></span>
                        <?php if (!empty($r['is_trx_validated'])): ?>
                          <i class="bi bi-check-circle-fill text-success" title="Validated"></i>
                        <?php endif; ?>
                      </span>
                    <?php else: ?>
                      <span class="text-muted">Not provided</span>
                    <?php endif; ?>
                  </p>
                </div>

                <div class="col-md-6">
                  <div class="small text-muted">Submitted On</div>
                  <p><?php echo e(format_date($r['created_at'])); ?></p>
                </div>

                <div class="col-12">
                  <div class="small text-muted">
                    <?php echo $stat === 'rejected' ? 'Rejection Reason' : 'Remarks'; ?>
                  </div>
                  <p>
                    <?php echo $r['remarks']
                        ? e($r['remarks'])
                        : '<span class="text-muted">None</span>'; ?>
                  </p>
                </div>
              </div>
            <?php else: ?>
              <div class="text-muted">
                This entry is from partner data. Please submit your deposit details.
              </div>
            <?php endif; ?>

            <div class="d-flex gap-2 flex-wrap mt-2">
              <?php if ($attachments): ?>
                <button
                  type="button"
                  class="btn btn-outline-secondary btn-sm"
                  data-bs-toggle="modal"
                  data-bs-target="#attachmentsModal"
                  data-attachments='<?php echo $attachmentsPayload; ?>'
                  data-date="<?php echo e($r['service_date']); ?>">
                  <i class="bi bi-paperclip"></i> View Attachments (<?php echo count($attachments); ?>)
                </button>
              <?php endif; ?>

              <?php if ($stat === 'yet to submit'): ?>
                <a class="btn btn-danger btn-sm"
                   href="<?php echo base_url('driver/submit.php'); ?>?date=<?php echo urlencode($r['service_date']); ?>">
                  Submit Now
                </a>
              <?php elseif ($can_resubmit): ?>
                <a class="btn btn-danger btn-sm"
                   href="<?php echo base_url('driver/resubmit.php'); ?>?id=<?php echo $sid; ?>">
                  Resubmit
                </a>
              <?php else: ?>
                <button class="btn btn-outline-secondary btn-sm" disabled>Resubmit</button>
              <?php endif; ?>
            </div>

          </div>
        </div>
      </div>
    <?php endforeach; ?>
  </div>
<?php endif; ?>

<div id="no-filter-results" class="alert alert-light text-center" style="display:none;"></div>

<!-- Attachments Modal -->
<div class="modal fade" id="attachmentsModal" tabindex="-1" aria-labelledby="attachmentsModalLabel" aria-hidden="true">
  <div class="modal-dialog modal-dialog-centered modal-lg">
    <div class="modal-content">

      <div class="modal-header bg-light">
        <h5 class="modal-title" id="attachmentsModalLabel">
          <i class="bi bi-paperclip me-2"></i>Attachments
        </h5>
        <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
      </div>

      <div class="modal-body">
        <ul class="nav nav-tabs" id="attachmentTabs" role="tablist"></ul>
        <div class="tab-content" id="attachmentTabsContent"></div>
      </div>

      <div class="modal-footer bg-light">
        <button class="btn btn-secondary" data-bs-dismiss="modal">
          <i class="bi bi-x-circle me-1"></i>Close
        </button>
      </div>

    </div>
  </div>
</div>

<script>
document.addEventListener('DOMContentLoaded', function () {
  const modal = document.getElementById('attachmentsModal');
  if (!modal) return;

  const attachmentTabs = modal.querySelector('#attachmentTabs');
  const attachmentTabsContent = modal.querySelector('#attachmentTabsContent');
  const modalLabel = document.getElementById('attachmentsModalLabel');

  modal.addEventListener('show.bs.modal', event => {
    const trigger = event.relatedTarget;
    if (!trigger) return;

    const date = trigger.getAttribute('data-date') || '';
    let attachments = [];

    try {
      attachments = JSON.parse(trigger.getAttribute('data-attachments') || '[]');
    } catch (e) {
      attachments = [];
    }

    modalLabel.innerHTML = '<i class="bi bi-paperclip me-2"></i>Attachments for ' + date;
    attachmentTabs.innerHTML = '';
    attachmentTabsContent.innerHTML = '';

    if (attachments.length === 0) {
      attachmentTabsContent.innerHTML =
        '<div class="text-center text-muted p-5"><i class="bi bi-inbox fs-1 mb-3"></i><br>No attachments available.</div>';
      return;
    }

    attachments.forEach((attachment, index) => {
      const tabId = 'attachmenttab-' + index;
      const paneId = 'attachmentpane-' + index;
      const isActive = index === 0;

      attachmentTabs.insertAdjacentHTML('beforeend', `
        <li class="nav-item" role="presentation">
          <button class="nav-link ${isActive ? 'active' : ''}"
                  id="${tabId}"
                  data-bs-toggle="tab"
                  data-bs-target="#${paneId}"
                  type="button"
                  role="tab"
                  aria-controls="${paneId}"
                  aria-selected="${isActive}">
            <i class="bi bi-file-earmark me-1"></i>${attachment.label}
          </button>
        </li>
      `);

      const isImage = /\.(jpg|jpeg|png|gif|webp)$/i.test(attachment.url);
const content = isImage
  ? `<img src="${attachment.url}" class="file-preview-img">`
  : `<iframe src="${attachment.url}" class="file-preview-frame"></iframe>`;


      attachmentTabsContent.insertAdjacentHTML('beforeend', `
        <div class="tab-pane fade ${isActive ? 'show active' : ''}"
             id="${paneId}"
             role="tabpanel"
             aria-labelledby="${tabId}">
          <div class="pt-3 text-center">${content}</div>
        </div>
      `);
    });
  });

  // Status filter logic
  const filterContainer = document.getElementById('status-filter');
  if (filterContainer) {
    const filterButtons = filterContainer.querySelectorAll('button');
    filterButtons.forEach(button => {
      button.addEventListener('click', function () {
        filterButtons.forEach(btn => {
          btn.classList.remove('btn-dark', 'active');
          btn.classList.add('btn-outline-secondary');
        });

        this.classList.add('btn-dark', 'active');
        this.classList.remove('btn-outline-secondary');

        const status = this.getAttribute('data-status');
        let visible = 0;

        document.querySelectorAll('.history-item').forEach(item => {
          if (status === 'all' || item.getAttribute('data-status') === status) {
            item.style.display = '';
            visible++;
          } else {
            item.style.display = 'none';
          }
        });

        const noResults = document.getElementById('no-filter-results');
        if (visible === 0) {
          noResults.textContent = 'No items match the selected filter.';
          noResults.style.display = 'block';
        } else {
          noResults.style.display = 'none';
        }
      });
    });
  }
});
</script>

<?php require_once __DIR__ . '/../../includes/driver_ui_end.php'; ?>
