<?php
// public/driver/kyd_manage.php

require_once __DIR__ . '/../../includes/driver_bootstrap.php';

$driver_id = get_current_driver_id();
$driver_sess = $_SESSION['driver'] ?? [];
$partner_id = $driver_sess['partner_id'] ?? 'DRIVER';

if (empty($_SESSION['csrf'])) $_SESSION['csrf'] = bin2hex(random_bytes(16));
$csrf = $_SESSION['csrf'];

// --- 1. CONFIGURATION (REMOVED SELFIE & EXTRAS) ---
$DOC_CONFIG = [
    'civil_id' => [
        'title' => 'Civil ID Card', 'icon' => 'bi-person-badge', 'mandatory' => true,
        'has_number' => true, 'has_expiry' => true, 'sides' => ['front', 'back']
    ],
    'license' => [
        'title' => 'Driving License', 'icon' => 'bi-card-heading', 'mandatory' => true,
        'has_number' => true, 'has_expiry' => true, 'sides' => ['front', 'back'],
        'extra_option' => 'same_as_civil'
    ],
    'vehicle_reg' => [
        'title' => 'Vehicle Registration', 'icon' => 'bi-car-front', 'mandatory' => true,
        'has_number' => true, 'has_expiry' => true, 'sides' => ['front', 'back']
    ]
];

// --- 2. HELPERS ---
function get_file_url(?string $path) {
    if (empty($path)) return '';
    if (preg_match('~^https?://~i', $path)) return $path;
    return file_url($path);
}

function get_doc_type_id(mysqli $conn, string $code) {
    $stmt = $conn->prepare("SELECT id FROM document_types WHERE code=? LIMIT 1");
    $stmt->bind_param('s', $code);
    $stmt->execute();
    $id = (int)($stmt->get_result()->fetch_column() ?? 0);
    if ($id === 0) {
        $name = ucwords(str_replace('_', ' ', $code));
        $stmt = $conn->prepare("INSERT INTO document_types (name, code, is_active) VALUES (?, ?, 1)");
        $stmt->bind_param('ss', $name, $code);
        $stmt->execute();
        $id = $conn->insert_id;
    }
    return $id;
}

function upsert_doc_set(mysqli $conn, int $driver_id, string $key, ?string $num, ?string $exp) {
    $stmt = $conn->prepare("SELECT id FROM driver_doc_sets WHERE driver_id=? AND group_key=? LIMIT 1");
    $stmt->bind_param('is', $driver_id, $key);
    $stmt->execute();
    $sid = $stmt->get_result()->fetch_column();

    if ($sid) {
        if ($num !== null || $exp !== null) {
            $stmt = $conn->prepare("UPDATE driver_doc_sets SET doc_number=COALESCE(?, doc_number), expiry_date=COALESCE(?, expiry_date) WHERE id=?");
            $stmt->bind_param('ssi', $num, $exp, $sid);
            $stmt->execute();
        }
        return $sid;
    } else {
        $num = $num ?? ''; $exp = $exp ?? '';
        $stmt = $conn->prepare("INSERT INTO driver_doc_sets (driver_id, group_key, doc_number, expiry_date, status) VALUES (?, ?, ?, ?, 'pending')");
        $stmt->bind_param('isss', $driver_id, $key, $num, $exp);
        $stmt->execute();
        return $conn->insert_id;
    }
}

function save_doc_file(mysqli $conn, int $driver_id, int $set_id, int $type_id, string $side, array $file, string $partner_id, string $group_key) {
    if ($file['error'] !== UPLOAD_ERR_OK) return "Upload error: " . $file['error'];
    $ext = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
    if (!in_array($ext, ['jpg','jpeg','png','pdf','webp'])) return "Invalid file type.";

    $y = date('Y'); $m = date('m');
    // Store in /storage/docs/ instead of /public/uploads/docs/
    $rel_path = "{$partner_id}/{$group_key}/{$y}/{$m}";
    $abs_root = STORAGE_DIR . '/docs/' . $rel_path;
    if (!is_dir($abs_root)) @mkdir($abs_root, 0755, true);

    $fname = $side . '_' . time() . '_' . bin2hex(random_bytes(3)) . '.' . $ext;
    if (!move_uploaded_file($file['tmp_name'], $abs_root . '/' . $fname)) return "Server write error.";

    // Store as storage-relative path for DB
    $web_path = 'storage/docs/' . $rel_path . '/' . $fname;

    $stmt = $conn->prepare("SELECT id FROM driver_documents WHERE doc_set_id=? AND side=? LIMIT 1");
    $stmt->bind_param('is', $set_id, $side);
    $stmt->execute();
    $existing_id = $stmt->get_result()->fetch_column();

    if ($existing_id) {
        $stmt = $conn->prepare("UPDATE driver_documents SET file_path=?, status='pending' WHERE id=?");
        $stmt->bind_param('si', $web_path, $existing_id);
    } else {
        $stmt = $conn->prepare("INSERT INTO driver_documents (driver_id, doc_set_id, doc_type_id, file_path, side, status) VALUES (?, ?, ?, ?, ?, 'pending')");
        $stmt->bind_param('iiiss', $driver_id, $set_id, $type_id, $web_path, $side);
    }
    $stmt->execute();
    $conn->query("UPDATE driver_doc_sets SET status='pending' WHERE id=$set_id");
    return null;
}

// --- 3. SUBMIT HANDLING ---
$msg = ''; $msg_type = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['csrf']) && hash_equals($csrf, $_POST['csrf'])) {
    $group = $_POST['group_key'] ?? '';

    if (array_key_exists($group, $DOC_CONFIG)) {
        $cfg = $DOC_CONFIG[$group];
        $type_id = get_doc_type_id($conn, $group);

        $num = !empty($_POST['doc_number']) ? trim($_POST['doc_number']) : null;
        if ($group === 'license' && !empty($_POST['same_as_civil'])) {
            $q = $conn->query("SELECT doc_number FROM driver_doc_sets WHERE driver_id=$driver_id AND group_key='civil_id'");
            if ($row = $q->fetch_row()) $num = $row[0];
        }

        $exp = !empty($_POST['expiry_date']) ? trim($_POST['expiry_date']) : null;
        $set_id = upsert_doc_set($conn, $driver_id, $group, $num, $exp);

        $errors = [];
        foreach ($cfg['sides'] as $side) {
            $input_name = 'file_' . $side;
            if (!empty($_FILES[$input_name]['name'])) {
                $err = save_doc_file($conn, $driver_id, $set_id, $type_id, $side, $_FILES[$input_name], $partner_id, $group);
                if ($err) $errors[] = ucfirst($side) . ": $err";
            }
        }

        if (empty($errors)) {
            $msg = "Saved successfully.";
            $msg_type = "success";
        } else {
            $msg = implode(', ', $errors);
            $msg_type = "danger";
        }
    }
}

// --- 4. FETCH DATA ---
$data = [];
$stmt = $conn->prepare("SELECT s.group_key, s.doc_number, s.expiry_date, s.status, d.side, d.file_path FROM driver_doc_sets s LEFT JOIN driver_documents d ON d.doc_set_id = s.id WHERE s.driver_id = ?");
$stmt->bind_param('i', $driver_id);
$stmt->execute();
$res = $stmt->get_result();

while ($r = $res->fetch_assoc()) {
    $k = $r['group_key'];
    if (!isset($data[$k])) $data[$k] = ['number' => $r['doc_number'], 'expiry' => $r['expiry_date'], 'status' => $r['status'], 'front' => null, 'back' => null];
    if ($r['side']) $data[$k][$r['side']] = $r['file_path'];
}

require_once __DIR__ . '/../../includes/driver_ui_start.php';
?>

<div class="container pb-5">
    <div class="row justify-content-center">
        <div class="col-lg-6">

            <div class="mb-3">
                <h4 class="fw-bold">Upload Documents</h4>
                <p class="text-muted small">Tap a section to open details.</p>
            </div>

            <?php if ($msg): ?>
                <div class="alert alert-<?= $msg_type ?> alert-dismissible fade show mb-4 border-0 shadow-sm">
                    <i class="bi <?= $msg_type=='success'?'bi-check-circle-fill':'bi-exclamation-triangle-fill' ?> me-2"></i>
                    <?= htmlspecialchars($msg) ?>
                    <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                </div>
            <?php endif; ?>

            <div class="accordion" id="docAccordion">
                <?php foreach ($DOC_CONFIG as $key => $cfg):
                    $d = $data[$key] ?? [];
                    $st = strtolower($d['status'] ?? 'pending');
                    $has_front = !empty($d['front']);

                    $badgeClass = 'bg-light text-secondary border';
                    $badgeText = 'Missing';

                    if ($st === 'approved') {
                        $badgeClass = 'bg-success text-white'; $badgeText = 'Approved';
                    } elseif ($st === 'rejected') {
                        $badgeClass = 'bg-danger text-white'; $badgeText = 'Rejected';
                    } elseif ($has_front) {
                        $badgeClass = 'bg-warning text-dark'; $badgeText = 'In Review';
                    }
                ?>

                <div class="card shadow-sm border-0 mb-3 overflow-hidden">
                    <div class="card-header bg-white border-0 p-0" id="head-<?= $key ?>">
                        <button class="btn w-100 text-start d-flex justify-content-between align-items-center p-3" type="button" data-bs-toggle="collapse" data-bs-target="#c-<?= $key ?>" aria-expanded="false">
                            <div class="d-flex align-items-center">
                                <i class="bi <?= $cfg['icon'] ?> fs-4 me-3 text-secondary"></i>
                                <div>
                                    <div class="fw-bold text-dark"><?= $cfg['title'] ?> <?php if($cfg['mandatory']): ?><span class="text-danger">*</span><?php endif; ?></div>
                                </div>
                            </div>
                            <span class="badge <?= $badgeClass ?> rounded-pill"><?= $badgeText ?></span>
                        </button>
                    </div>

                    <div id="c-<?= $key ?>" class="collapse" data-bs-parent="#docAccordion">
                        <div class="card-body bg-light border-top">
                            <form method="post" enctype="multipart/form-data">
                                <input type="hidden" name="csrf" value="<?= $csrf ?>">
                                <input type="hidden" name="group_key" value="<?= $key ?>">

                                <?php if ($cfg['has_number'] || $cfg['has_expiry']): ?>
                                    <div class="bg-white p-3 rounded border-0 shadow-sm mb-3">
                                        <div class="row g-3">
                                            <?php if ($cfg['has_number']): ?>
                                                <div class="col-12">
                                                    <label class="form-label small text-muted fw-bold">Document Number</label>
                                                    <input type="text" name="doc_number" class="form-control form-control-sm" value="<?= htmlspecialchars($d['number'] ?? '') ?>">
                                                </div>
                                            <?php endif; ?>
                                            <?php if ($cfg['has_expiry']): ?>
                                                <div class="col-12"><label class="form-label small text-muted fw-bold">Expiry Date</label><input type="date" name="expiry_date" class="form-control form-control-sm" value="<?= htmlspecialchars($d['expiry'] ?? '') ?>"></div>
                                            <?php endif; ?>
                                        </div>
                                    </div>
                                <?php endif; ?>

                                <div class="row g-2">
                                    <?php foreach ($cfg['sides'] as $side): ?>
                                        <div class="col-6">
                                            <div class="bg-white p-2 rounded border text-center h-100 position-relative">
                                                <label class="small text-muted fw-bold d-block mb-2"><?= ucfirst($side) ?></label>
                                                <?php if(!empty($d[$side])): ?>
                                                    <a href="<?= get_file_url($d[$side]) ?>" target="_blank">
                                                        <img src="<?= get_file_url($d[$side]) ?>" class="img-fluid rounded border mb-2" style="height: 80px; object-fit: cover;">
                                                    </a>
                                                <?php else: ?>
                                                    <div class="py-3 text-muted"><i class="bi bi-cloud-arrow-up fs-3 opacity-25"></i></div>
                                                <?php endif; ?>
                                                <input type="file" name="file_<?= $side ?>" class="form-control form-control-sm" style="font-size:0.8rem;">
                                        </div>
                                    </div>
                                    <?php endforeach; ?>
                                </div>

                                <div class="d-grid mt-3">
                                    <button class="btn btn-dark shadow-sm">Save Changes</button>
                                </div>
                            </form>
                        </div>
                    </div>
                </div>
            <?php endforeach; ?>
            </div>
        </div>
    </div>
</div>

<script>
document.querySelectorAll('form').forEach(f => {
    f.addEventListener('submit', e => {
        const b = e.submitter;
        if(b) { b.innerHTML='<span class="spinner-border spinner-border-sm"></span>'; b.disabled=true; }
    })
});
</script>

<?php require_once __DIR__ . '/../../includes/driver_ui_end.php'; ?>
