<?php
// public/driver/login.php
require_once __DIR__ . '/../../includes/driver_auth.php';
require_once __DIR__ . '/../../includes/helpers.php';
require_once __DIR__ . '/../../includes/settings.php';
require_once __DIR__ . '/../../config/db.php';

$err = '';
if (empty($_SESSION['csrf'])) $_SESSION['csrf'] = bin2hex(random_bytes(16));
$csrf = $_SESSION['csrf'];

// Fetch branding settings
$appName = app_setting('branding.app_name', 'Hisbuu');
$loginLogo = app_setting('branding.driver_login_logo', '');
if ($loginLogo !== '' && strpos($loginLogo, 'storage/') === 0) {
    $loginLogo = file_url($loginLogo);
}

// Redirect if already logged in
redirect_driver_if_logged_in();

// Handle login
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['csrf']) && hash_equals($csrf, $_POST['csrf'])) {
  $uuid = trim($_POST['uuid'] ?? '');
  $pass = (string)($_POST['password'] ?? '');

  if ($uuid === '' || $pass === '') {
    $err = 'Driver ID and password are required.';
  } else {
    $stmt = $conn->prepare("SELECT id, partner_id, first_name, last_name, login_password, status FROM drivers WHERE partner_id=? LIMIT 1");
    $stmt->bind_param('s', $uuid);
    $stmt->execute();
    $row = $stmt->get_result()->fetch_assoc();

    if (!$row || strtolower($row['status']) !== 'active') {
      $err = 'Invalid credentials.';
    } elseif (!password_verify($pass, $row['login_password'])) {
      $err = 'Invalid credentials.';
    } else {
      $_SESSION['driver'] = [
        'id'   => (int)$row['id'],
        'partner_id' => $row['partner_id'],
        'uuid' => $row['partner_id'],
        'name' => trim(($row['first_name'] ?? '') . ' ' . ($row['last_name'] ?? '')),
      ];
      session_regenerate_id(true);
      header('Location: dashboard.php');
      exit;
    }
  }
}
?>
<!doctype html>
<html lang="en">
<head>
  <meta charset="utf-8">
  <title>Driver Login | <?php echo e($appName); ?></title>
  <meta name="viewport" content="width=device-width, initial-scale=1, maximum-scale=1, user-scalable=no">
  <link rel="manifest" href="manifest.json">
  <meta name="apple-mobile-web-app-capable" content="yes">
  <meta name="apple-mobile-web-app-status-bar-style" content="black-translucent">
  <meta name="theme-color" content="#dc3545">
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
  <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.3/font/bootstrap-icons.min.css" rel="stylesheet">
  <style>
    :root {
      --red: #dc3545;
      --red-dark: #b02a37;
      --red-soft: #fee2e2;
      --gray-50: #f8fafc;
      --gray-100: #f1f5f9;
      --gray-400: #94a3b8;
      --gray-500: #64748b;
      --gray-600: #475569;
      --gray-900: #0f172a;
    }

    * {
      -webkit-tap-highlight-color: transparent;
    }

    body {
      background: linear-gradient(180deg, var(--red) 0%, var(--red-dark) 100%);
      min-height: 100vh;
      min-height: 100dvh;
      font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
    }

    .login-container {
      min-height: 100vh;
      min-height: 100dvh;
      display: flex;
      flex-direction: column;
    }

    /* Top branding area */
    .login-header {
      padding: 40px 24px 30px;
      text-align: center;
      color: #fff;
      flex-shrink: 0;
    }

    .logo-wrapper {
      margin-bottom: 16px;
    }

    .logo-wrapper img {
      max-height: 48px;
      max-width: 160px;
      object-fit: contain;
    }

    .logo-circle {
      width: 64px;
      height: 64px;
      background: rgba(255,255,255,0.15);
      border-radius: 50%;
      display: inline-flex;
      align-items: center;
      justify-content: center;
      font-size: 28px;
      font-weight: 700;
      color: #fff;
      backdrop-filter: blur(10px);
    }

    .login-header h1 {
      font-size: 22px;
      font-weight: 600;
      margin: 12px 0 6px;
    }

    .login-header p {
      font-size: 14px;
      opacity: 0.85;
      margin: 0;
    }

    /* Card area */
    .login-card-wrapper {
      flex: 1;
      background: var(--gray-50);
      border-radius: 24px 24px 0 0;
      padding: 28px 20px 40px;
      display: flex;
      flex-direction: column;
    }

    .login-card {
      background: #fff;
      border-radius: 16px;
      padding: 24px 20px;
      box-shadow: 0 1px 3px rgba(0,0,0,0.06), 0 4px 12px rgba(0,0,0,0.04);
      max-width: 400px;
      width: 100%;
      margin: 0 auto;
    }

    .form-label {
      font-size: 13px;
      font-weight: 500;
      color: var(--gray-600);
      margin-bottom: 6px;
    }

    .form-control {
      border: 1.5px solid #e2e8f0;
      border-radius: 12px;
      padding: 14px 16px;
      font-size: 16px;
      transition: border-color 0.15s, box-shadow 0.15s;
    }

    .form-control:focus {
      border-color: var(--red);
      box-shadow: 0 0 0 3px var(--red-soft);
    }

    .form-control::placeholder {
      color: var(--gray-400);
    }

    .input-group-icon {
      position: relative;
    }

    .input-group-icon i {
      position: absolute;
      left: 16px;
      top: 50%;
      transform: translateY(-50%);
      color: var(--gray-400);
      font-size: 18px;
      pointer-events: none;
    }

    .input-group-icon .form-control {
      padding-left: 48px;
    }

    .btn-login {
      width: 100%;
      background: var(--red);
      border: none;
      border-radius: 12px;
      padding: 14px 20px;
      font-size: 16px;
      font-weight: 600;
      color: #fff;
      transition: background 0.15s, transform 0.1s;
    }

    .btn-login:hover {
      background: var(--red-dark);
    }

    .btn-login:active {
      transform: scale(0.98);
    }

    .alert-error {
      background: var(--red-soft);
      border: none;
      border-radius: 12px;
      color: var(--red-dark);
      padding: 12px 16px;
      font-size: 14px;
      display: flex;
      align-items: center;
      gap: 10px;
    }

    .alert-error i {
      font-size: 18px;
    }

    /* Features section */
    .features {
      margin-top: auto;
      padding-top: 24px;
      max-width: 400px;
      width: 100%;
      margin-left: auto;
      margin-right: auto;
    }

    .features-title {
      font-size: 12px;
      text-transform: uppercase;
      letter-spacing: 0.5px;
      color: var(--gray-500);
      margin-bottom: 12px;
      font-weight: 600;
    }

    .feature-item {
      display: flex;
      align-items: center;
      gap: 12px;
      padding: 10px 0;
    }

    .feature-icon {
      width: 36px;
      height: 36px;
      background: var(--red-soft);
      border-radius: 10px;
      display: flex;
      align-items: center;
      justify-content: center;
      color: var(--red);
      font-size: 16px;
      flex-shrink: 0;
    }

    .feature-text {
      font-size: 13px;
      color: var(--gray-600);
      line-height: 1.4;
    }

    /* Footer */
    .login-footer {
      text-align: center;
      padding: 16px 20px 24px;
      font-size: 12px;
      color: var(--gray-500);
    }

    /* Desktop adjustments */
    @media (min-width: 768px) {
      .login-header {
        padding: 60px 24px 40px;
      }

      .logo-circle {
        width: 80px;
        height: 80px;
        font-size: 36px;
      }

      .login-header h1 {
        font-size: 26px;
      }

      .login-card-wrapper {
        border-radius: 32px 32px 0 0;
        padding: 40px 24px 60px;
      }

      .login-card {
        padding: 32px;
      }
    }

    /* Safe area for notched devices */
    @supports (padding-bottom: env(safe-area-inset-bottom)) {
      .login-card-wrapper {
        padding-bottom: calc(40px + env(safe-area-inset-bottom));
      }
    }
  </style>
</head>
<body>

<div class="login-container">
  <!-- Header with branding -->
  <div class="login-header">
    <div class="logo-wrapper">
      <?php if ($loginLogo): ?>
        <img src="<?php echo e($loginLogo); ?>" alt="<?php echo e($appName); ?>">
      <?php else: ?>
        <div class="logo-circle"><?php echo e(mb_substr($appName, 0, 1)); ?></div>
      <?php endif; ?>
    </div>
    <h1><?php echo e($appName); ?></h1>
    <p>Driver Portal</p>
  </div>

  <!-- Login form card -->
  <div class="login-card-wrapper">
    <div class="login-card">
      <?php if ($err): ?>
        <div class="alert-error mb-3">
          <i class="bi bi-exclamation-circle"></i>
          <span><?php echo e($err); ?></span>
        </div>
      <?php endif; ?>

      <form method="post">
        <input type="hidden" name="csrf" value="<?php echo e($csrf); ?>">

        <div class="mb-3">
          <label class="form-label">Driver ID</label>
          <div class="input-group-icon">
            <i class="bi bi-person"></i>
            <input type="text" name="uuid" class="form-control" placeholder="e.g. TB00000123" required autocomplete="username">
          </div>
        </div>

        <div class="mb-4">
          <label class="form-label">Password</label>
          <div class="input-group-icon">
            <i class="bi bi-lock"></i>
            <input type="password" name="password" class="form-control" placeholder="Enter your password" required autocomplete="current-password">
          </div>
        </div>

        <button type="submit" class="btn-login">
          Sign In
        </button>
      </form>
    </div>

    <!-- Features -->
    <div class="features">
      <div class="features-title">What you can do</div>
      <div class="feature-item">
        <div class="feature-icon"><i class="bi bi-clipboard-check"></i></div>
        <div class="feature-text">Submit daily orders and cash deposits</div>
      </div>
      <div class="feature-item">
        <div class="feature-icon"><i class="bi bi-camera"></i></div>
        <div class="feature-text">Upload screenshots and deposit slips</div>
      </div>
      <div class="feature-item">
        <div class="feature-icon"><i class="bi bi-calendar3"></i></div>
        <div class="feature-text">Track your monthly submission history</div>
      </div>
    </div>

    <!-- Footer -->
    <div class="login-footer">
      Forgot password? Contact your supervisor.
    </div>
  </div>
</div>

</body>
</html>
