<?php
// public/driver/reports.php
require_once __DIR__ . '/../../includes/driver_bootstrap.php';

$driver_id = get_current_driver_id();
$title = 'Deposit Report';

// --- Data Fetching ---

// Initialize arrays for last 12 months with 0 values
$months_deposits = [];
$months_orders = [];
for ($i = 11; $i >= 0; $i--) {
    $date = new DateTime("first day of -$i month");
    $month_key = $date->format('Y-m');
    $months_deposits[$month_key] = 0;
    $months_orders[$month_key] = 0;
}

// Query for actual approved deposits in the last 12 months
$twelve_months_ago = (new DateTime("first day of -11 month"))->format('Y-m-d');

$sql = "
  SELECT
    DATE_FORMAT(service_date, '%Y-%m') AS month,
    SUM(cash_deposited) AS total_deposit,
    SUM(orders_count) AS total_orders
  FROM submissions
  WHERE
    driver_id = ? AND
    status = 'approved' AND
    service_date >= ?
  GROUP BY month
  ORDER BY month ASC
  ";

$stmt = $conn->prepare($sql);
$stmt->bind_param('is', $driver_id, $twelve_months_ago);
$stmt->execute();
$result = $stmt->get_result();

while ($row = $result->fetch_assoc()) {
    if (isset($months_deposits[$row['month']])) {
        $months_deposits[$row['month']] = (float)$row['total_deposit'];
        $months_orders[$row['month']] = (int)$row['total_orders'];
    }
}
$stmt->close();

// Prepare data for Chart.js
$chart_labels = [];
foreach (array_keys($months_deposits) as $month_key) {
    $chart_labels[] = DateTime::createFromFormat('Y-m', $month_key)->format('M Y');
}
$chart_data_deposits = array_values($months_deposits);
$chart_data_orders = array_values($months_orders);

require_once __DIR__ . '/../../includes/driver_ui_start.php';
?>

<h5 class="mb-3">Monthly Deposit Report</h5>

<div class="card shadow-sm">
    <div class="card-body">
        <p class="text-muted">Total approved deposits over the last 12 months.</p>
        <canvas id="depositsChart"></canvas>
    </div>
</div>

<!-- Include Chart.js library -->
<script src="https://cdn.jsdelivr.net/npm/chart.js"></script>

<script>
document.addEventListener('DOMContentLoaded', function () {
    const ctx = document.getElementById('depositsChart').getContext('2d');
    new Chart(ctx, {
        type: 'bar', // Default type
        data: {
            labels: <?php echo json_encode($chart_labels); ?>,
            datasets: [{
                type: 'bar',
                label: 'Total Deposits (OMR)',
                data: <?php echo json_encode($chart_data_deposits); ?>,
                backgroundColor: 'rgba(220, 53, 69, 0.7)',
                borderColor: 'rgba(220, 53, 69, 1)',
                borderWidth: 1,
                yAxisID: 'yDeposits'
            }, {
                type: 'line',
                label: 'Total Orders',
                data: <?php echo json_encode($chart_data_orders); ?>,
                backgroundColor: 'rgba(13, 110, 253, 0.2)',
                borderColor: 'rgba(13, 110, 253, 1)',
                borderWidth: 2,
                fill: true,
                tension: 0.3,
                yAxisID: 'yOrders'
            }]
        },
        options: {
            responsive: true,
            scales: {
                yDeposits: {
                    type: 'linear',
                    position: 'left',
                    beginAtZero: true,
                    title: { display: true, text: 'Deposits (OMR)' }
                },
                yOrders: {
                    type: 'linear',
                    position: 'right',
                    beginAtZero: true,
                    title: { display: true, text: 'Orders' },
                    grid: { drawOnChartArea: false } // only show grid for left axis
                }
            },
            plugins: {
                legend: { display: true },
                tooltip: {
                    callbacks: {
                        label: function(context) {
                            let label = context.dataset.label || '';
                            if (label) {
                                label += ': ';
                            }
                            if (context.parsed.y !== null) {
                                label += new Intl.NumberFormat('en-OM', { style: 'currency', currency: 'OMR' }).format(context.parsed.y);
                            }
                            return label;
                        }
                    }
                }
            }
        }
    });
});
</script>

<?php
require_once __DIR__ . '/../../includes/driver_ui_end.php';
