<?php
// /public/driver/submit.php
require_once __DIR__ . '/../../includes/driver_bootstrap.php';
require_once __DIR__ . '/../../includes/uploads.php';

$driver_id = get_current_driver_id();
$csrf = get_csrf_token();

// fetch driver uuid + name
$stmtDU = $conn->prepare("
  SELECT partner_id, CONCAT(COALESCE(first_name,''),' ',COALESCE(last_name,'')) AS full_name
  FROM drivers WHERE id=? LIMIT 1
  ");
$stmtDU->bind_param('i', $driver_id);
$stmtDU->execute();
$rowDU = $stmtDU->get_result()->fetch_assoc();
if (!$rowDU) { http_response_code(403); exit('Driver not found'); }
$driver_uuid = $rowDU['partner_id'];
$driver_name = trim($rowDU['full_name'] ?? '');

$err = '';
$ok  = '';

// local dates
$today      = today_local();
$yesterday  = yesterday_local();
$yesterdayStr = $yesterday->format('Y-m-d');

// Pre-fill from partner data if available
$pre_orders = '';
$pre_amount = '';
$target_date = $_GET['date'] ?? '';
if ($target_date && preg_match('/^\d{4}-\d{2}-\d{2}$/', $target_date)) {
  $pd = $conn->prepare("SELECT orders_count, amount_total, total_orders, total_amount FROM partner_daily WHERE driver_id=? AND service_date=? LIMIT 1");
  $pd->bind_param('is', $driver_id, $target_date);
  $pd->execute();
  if ($r = $pd->get_result()->fetch_assoc()) {
    $pre_orders = $r['orders_count'] ?: $r['total_orders'];
    $amt = $r['amount_total'] != 0 ? $r['amount_total'] : $r['total_amount'];
    if ($amt) $pre_amount = number_format((float)$amt, 3, '.', '');
  }
}

// handle submit
if (is_valid_post()) {
  $service_date = trim($_POST['service_date'] ?? '');
  $orders_raw   = trim($_POST['orders_count'] ?? '');
  $cash_raw     = trim($_POST['cash_deposited'] ?? '');
  $trx_no       = trim($_POST['trx_no'] ?? '');
  $remarks      = trim($_POST['remarks'] ?? '');

  // date validation
  if ($service_date === '' || !preg_match('/^\d{4}-\d{2}-\d{2}$/', $service_date)) {
    $err = 'Select a valid date.';
  } else {
    $d = DateTime::createFromFormat('Y-m-d', $service_date);
    if (!$d) {
      $err = 'Invalid date format.';
    } elseif ($d >= $today) {
      $err = 'You cannot submit for today or future dates.';
    }
  }

  // numeric validation
  if (!$err) {
    if ($orders_raw === '' || !ctype_digit($orders_raw)) {
      $err = 'Orders must be a whole number.';
    } elseif ($cash_raw === '' || !preg_match('/^\d+(\.\d{1,3})?$/', $cash_raw)) {
      $err = 'Cash amount must be a number with up to 3 decimals.';
    }
  }

  // one per day
  if (!$err) {
    $chk = $conn->prepare("SELECT id FROM submissions WHERE driver_id=? AND service_date=? LIMIT 1");
    $chk->bind_param('is', $driver_id, $service_date);
    $chk->execute();
    if ($chk->get_result()->fetch_column()) {
      $err = 'You already submitted for this date. Use Resubmit from History.';
    }
  }

  // files
  $path_orders = $path_dep1 = $path_dep2 = '';
  if (!$err) {
    list($absDir, $relDir) = build_submission_dir($driver_uuid, $service_date);

    // required order screenshot
    if (!isset($_FILES['ss_orders']) || $_FILES['ss_orders']['error'] !== UPLOAD_ERR_OK) {
      $err = 'Order screenshot is required.';
    } elseif (!allowed_image($_FILES['ss_orders']['tmp_name'], $_FILES['ss_orders']['name'])) {
      $err = 'Order screenshot must be an image.';
    } else {
      $path_orders = save_submission_file($_FILES['ss_orders'], $absDir, $relDir, 'orders');
      if ($path_orders === '') $err = 'Failed to save order screenshot.';
    }

    // optional deposits
    if (!$err && isset($_FILES['ss_deposit_1']) && $_FILES['ss_deposit_1']['error'] === UPLOAD_ERR_OK) {
      if (!allowed_image($_FILES['ss_deposit_1']['tmp_name'], $_FILES['ss_deposit_1']['name'])) {
        $err = 'Deposit 1 must be an image.';
      } else {
        $path_dep1 = save_submission_file($_FILES['ss_deposit_1'], $absDir, $relDir, 'deposit1');
        if ($path_dep1 === '') $err = 'Failed to save deposit 1.';
      }
    }
    if (!$err && isset($_FILES['ss_deposit_2']) && $_FILES['ss_deposit_2']['error'] === UPLOAD_ERR_OK) {
      if (!allowed_image($_FILES['ss_deposit_2']['tmp_name'], $_FILES['ss_deposit_2']['name'])) {
        $err = 'Deposit 2 must be an image.';
      } else {
        $path_dep2 = save_submission_file($_FILES['ss_deposit_2'], $absDir, $relDir, 'deposit2');
        if ($path_dep2 === '') $err = 'Failed to save deposit 2.';
      }
    }
  }

  // insert
  if (!$err) {
    $orders = (int)$orders_raw;
    $cash   = (float)$cash_raw;

    $ins = $conn->prepare("
      INSERT INTO submissions
        (driver_id, service_date, orders_count, cash_deposited, ss_orders, ss_deposit_1, ss_deposit_2, trx_no, remarks, status, created_at, updated_at)
      VALUES
        (?, ?, ?, ?, ?, ?, ?, ?, ?, 'pending', NOW(), NOW())
    ");
    $ins->bind_param(
      'isidsssss',
      $driver_id, $service_date, $orders, $cash,
      $path_orders, $path_dep1, $path_dep2, $trx_no, $remarks
    );

    if ($ins->execute()) {
      header('Location: dashboard.php?ok=1');
      exit;
    }
    $err = 'Failed to create submission.';
  }
}

// UI
$title = 'Create submission';
require_once __DIR__ . '/../../includes/driver_ui_start.php';
?>
  <div class="d-flex justify-content-between align-items-center mb-3">
    <h5 class="mb-0">Create a submission</h5>
    <a class="btn btn-outline-secondary btn-sm" href="dashboard.php">Back</a>
  </div>

  <?php if ($err): ?><div class="alert alert-danger"><?php echo $err; ?></div><?php endif; ?>

  <form method="post" enctype="multipart/form-data">
    <input type="hidden" name="csrf" value="<?php echo e($csrf); ?>">

    <!-- Main details -->
    <div class="card mb-3">
      <div class="card-body">
        <div class="row g-3">
          <div class="col-12 col-md-6">
            <label class="form-label">Date of work</label>
            <input type="date" name="service_date" class="form-control" max="<?php echo e($yesterdayStr); ?>" value="<?php echo e($_GET['date'] ?? ''); ?>" <?php echo !empty($_GET['date']) ? 'readonly' : 'required'; ?>>
            <div class="form-text">You cannot submit for today or future dates.</div>
          </div>
          <div class="col-12 col-md-6">
            <label class="form-label">Total orders</label>
            <input type="number" name="orders_count" class="form-control" min="0" step="1" required value="<?php echo e($_POST['orders_count'] ?? $pre_orders); ?>">
            <?php if($pre_orders !== ''): ?>
              <div class="form-text text-success"><i class="bi bi-info-circle"></i> Partner data: <?php echo e($pre_orders); ?></div>
            <?php endif; ?>
          </div>
          <div class="col-12">
            <label class="form-label">Total cash deposited (OMR)</label>
            <input type="text" name="cash_deposited" class="form-control" placeholder="0.000" pattern="^\d+(\.\d{1,3})?$" required value="<?php echo e($_POST['cash_deposited'] ?? $pre_amount); ?>">
            <?php if($pre_amount !== ''): ?>
              <div class="form-text text-success"><i class="bi bi-info-circle"></i> Partner data: <?php echo e($pre_amount); ?></div>
            <?php endif; ?>
            <div class="form-text">E.g. 123.450</div>
          </div>
        </div>
      </div>
    </div>

    <!-- Attachments -->
    <div class="card mb-3">
      <div class="card-header">Attachments</div>
      <div class="card-body">
        <div class="row g-3">
          <div class="col-12 col-md-4">
            <label class="form-label">Order screenshot</label>
            <input type="file" name="ss_orders" class="form-control" accept="image/*" required>
            <div class="form-text small">Required.</div>
          </div>
          <div class="col-12 col-md-4">
            <label class="form-label">Deposit slip 1</label>
            <input type="file" name="ss_deposit_1" class="form-control" accept="image/*">
          </div>
          <div class="col-12 col-md-4">
            <label class="form-label">Deposit slip 2</label>
            <input type="file" name="ss_deposit_2" class="form-control" accept="image/*">
          </div>
        </div>
      </div>
    </div>

    <!-- Optional Info -->
    <div class="card mb-3">
      <div class="card-header">
        <a href="#optional-info" data-bs-toggle="collapse" class="text-decoration-none text-dark">
          Optional Information
        </a>
      </div>
      <div class="collapse show" id="optional-info">
        <div class="card-body">
          <div class="row g-3">
            <div class="col-12 col-md-6">
              <label class="form-label">Bank TRX No.</label>
              <input type="text" name="trx_no" class="form-control" maxlength="64" placeholder="e.g., LFT200300400">
            </div>
            <div class="col-12">
              <label class="form-label">Remarks</label>
              <textarea name="remarks" class="form-control" rows="2" maxlength="500"></textarea>
            </div>
          </div>
        </div>
      </div>
    </div>

    <div class="d-grid">
      <button class="btn btn-danger btn-lg">Submit for Approval</button>
    </div>
  </form>

<?php
require_once __DIR__ . '/../../includes/driver_ui_end.php';
