<?php
require_once __DIR__ . '/../../includes/driver_bootstrap.php';

$driver_id = get_current_driver_id();

function badge_html($s) {
  $s = strtolower((string)$s);
  if ($s === 'approved') return '<span class="badge text-bg-success">Approved</span>';
  if ($s === 'rejected') return '<span class="badge text-bg-danger">Rejected</span>';
  return '<span class="badge text-bg-warning text-dark">Pending</span>';
}

// Fetch document sets for this driver (avoid get_result dependency)
$doc_labels = [
  'civil_id'    => 'ID Card',
  'license'     => 'Driving License',
  'vehicle_reg' => 'Vehicle Registration',
];

$stmt = $conn->prepare("
  SELECT id, group_key, status, remarks, updated_at
  FROM driver_doc_sets
  WHERE driver_id = ?
  ORDER BY group_key
");
$stmt->bind_param('i', $driver_id);
$stmt->execute();
$stmt->bind_result($set_id, $group_key, $status, $remarks, $updated_at);
$doc_sets = [];
while ($stmt->fetch()) {
  $doc_sets[$set_id] = [
    'id'         => $set_id,
    'group_key'  => $group_key,
    'status'     => $status,
    'remarks'    => $remarks,
    'updated_at' => $updated_at,
    'doc_name'   => $doc_labels[$group_key] ?? ucwords(str_replace('_', ' ', (string)$group_key)),
    'front_path' => null,
    'back_path'  => null,
  ];
}
$stmt->close();

// Fetch all files for these sets and map them
if (!empty($doc_sets)) {
  $set_ids = array_keys($doc_sets);
  $in_clause = implode(',', array_fill(0, count($set_ids), '?'));
  $types = str_repeat('i', count($set_ids));

  $stmt = $conn->prepare("SELECT doc_set_id, side, file_path FROM driver_documents WHERE doc_set_id IN ($in_clause)");
  $stmt->bind_param($types, ...$set_ids);
  $stmt->execute();
  $stmt->bind_result($doc_set_id, $side, $file_path);
  while ($stmt->fetch()) {
    $key = ($side === 'back') ? 'back_path' : 'front_path';
    if (isset($doc_sets[$doc_set_id])) $doc_sets[$doc_set_id][$key] = $file_path;
  }
  $stmt->close();
}

// Simple counts
$cnt = ['pending'=>0,'approved'=>0,'rejected'=>0];
foreach ($doc_sets as $d) {
  $k = strtolower($d['status'] ?? 'pending');
  if (isset($cnt[$k])) $cnt[$k]++;
}

// Page
?>
<!doctype html>
<html lang="en">
<head>
  <meta charset="utf-8">
  <title>My Documents | Hisbuu</title>
  <meta name="viewport" content="width=device-width, initial-scale=1">
</head>
<body class="bg-light">

<?php
$title = 'My Documents';
require_once __DIR__ . '/../../includes/driver_ui_start.php';
?>

<style>
  .doc-card { border: 1px solid #e5e7eb; }
  .doc-thumb { border: 1px dashed #ced4da; border-radius: 10px; background: #fdfdfd; }
  .doc-thumb img { max-height: 160px; object-fit: contain; width: 100%; }
  .doc-label { font-size: .8rem; color: #6c757d; }
  .pill { padding: .25rem .6rem; border-radius: 999px; font-size: .78rem; }
  @media (max-width: 576px) {
    .doc-card .btn { width: 100%; }
  }
</style>

<div class="d-flex flex-column flex-md-row justify-content-between align-items-start align-items-md-center mb-3">
  <div>
    <h5 class="mb-1">My Documents</h5>
    <div class="text-muted small">Upload, view and track your KYD documents.</div>
  </div>
  <div class="d-flex gap-2 mt-2 mt-md-0">
    <a class="btn btn-sm btn-outline-secondary d-flex align-items-center gap-1" href="?refresh=1">
      <i class="bi bi-arrow-clockwise"></i> Refresh
    </a>
    <a class="btn btn-sm btn-danger d-flex align-items-center gap-1" href="kyd_manage.php">
      <i class="bi bi-pencil-square"></i> Update Documents
    </a>
  </div>
</div>

<!-- Summary Counts -->
<div class="card shadow-sm mb-4">
  <div class="card-body">
    <div class="row text-center gy-3">
      <div class="col-4 border-end">
        <div class="text-muted text-uppercase small">Pending</div>
        <div class="fs-5 fw-semibold"><?php echo (int)$cnt['pending']; ?></div>
      </div>
      <div class="col-4 border-end">
        <div class="text-muted text-uppercase small">Approved</div>
        <div class="fs-5 fw-semibold text-success"><?php echo (int)$cnt['approved']; ?></div>
      </div>
      <div class="col-4">
        <div class="text-muted text-uppercase small">Rejected</div>
        <div class="fs-5 fw-semibold text-danger"><?php echo (int)$cnt['rejected']; ?></div>
      </div>
    </div>
  </div>
</div>

<?php if (!$doc_sets): ?>
  <div class="alert alert-info">
    You have not uploaded any documents yet. Go to KYD to upload.
  </div>
<?php else: ?>
  <div class="row row-cols-1 row-cols-md-2 row-cols-lg-3 g-4">
    <?php foreach ($doc_sets as $set): ?>
    <div class="col">
      <div class="card doc-card h-100 shadow-sm">
        <div class="card-body d-flex flex-column gap-3">
          <div class="d-flex justify-content-between align-items-start">
            <div>
              <div class="fw-semibold"><?php echo htmlspecialchars($set['doc_name']); ?></div>
              <div class="doc-label">Last update: <?php echo htmlspecialchars($set['updated_at']); ?></div>
            </div>
            <div><?php echo badge_html($set['status']); ?></div>
          </div>

          <div class="doc-thumbs row row-cols-1 row-cols-sm-2 g-2">
            <div class="col">
              <div class="doc-label mb-1">Front</div>
              <?php if (!empty($set['front_path'])): ?>
                <a href="<?php echo htmlspecialchars(file_url($set['front_path'])); ?>" target="_blank" rel="noopener" class="doc-thumb d-block p-2 text-center">
                  <img src="<?php echo htmlspecialchars(file_url($set['front_path'])); ?>" alt="front side">
                </a>
              <?php else: ?>
                <div class="doc-thumb d-flex align-items-center justify-content-center p-3 text-muted small">Not uploaded</div>
              <?php endif; ?>
            </div>
            <div class="col">
              <div class="doc-label mb-1">Back</div>
              <?php if (!empty($set['back_path'])): ?>
                <a href="<?php echo htmlspecialchars(file_url($set['back_path'])); ?>" target="_blank" rel="noopener" class="doc-thumb d-block p-2 text-center">
                  <img src="<?php echo htmlspecialchars(file_url($set['back_path'])); ?>" alt="back side">
                </a>
              <?php else: ?>
                <div class="doc-thumb d-flex align-items-center justify-content-center p-3 text-muted small">Not uploaded</div>
              <?php endif; ?>
            </div>
          </div>

          <?php if ($set['remarks']): ?>
            <div class="alert alert-warning py-2 mb-0 small">
              <strong>Remarks:</strong> <?php echo htmlspecialchars($set['remarks']); ?>
            </div>
          <?php endif; ?>
        </div>
        <div class="card-footer bg-white">
          <div class="d-grid d-sm-flex gap-2">
            <?php if (strtolower($set['status']) === 'rejected'): ?>
              <a class="btn btn-sm btn-warning" href="kyd_manage.php#<?php echo e($set['group_key']); ?>">Fix and re-upload</a>
            <?php else: ?>
              <a class="btn btn-sm btn-outline-secondary" href="kyd_manage.php#<?php echo e($set['group_key']); ?>">View/Update</a>
            <?php endif; ?>
          </div>
        </div>
      </div>
    </div>
    <?php endforeach; ?>
  </div>
<?php endif; ?>

<?php
require_once __DIR__ . '/../../includes/driver_ui_end.php';
?>
</body>
</html>
