<?php
/**
 * serve.php - Secure file server for files stored outside public web root
 *
 * Serves files from /storage/ directory with authentication and access control.
 * Usage: serve.php?type=submissions&path=YYYY/MM/UUID/YYYY-MM-DD/file.jpg
 *        serve.php?type=docs&path=UUID/group_key/YYYY/MM/file.jpg
 *        serve.php?type=branding&path=admin_logo.png (public, no auth)
 */

declare(strict_types=1);

require_once __DIR__ . '/../includes/constants.php';
require_once __DIR__ . '/../includes/session_config.php';

// Start session if not already started
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

// Get parameters first to check if branding (public access)
$type = $_GET['type'] ?? '';
$path = $_GET['path'] ?? '';

// Validate type
$allowed_types = ['submissions', 'docs', 'branding'];
if (!in_array($type, $allowed_types, true)) {
    http_response_code(400);
    exit('Invalid type');
}

// Determine user type and authenticate
$user_type = null;
$user_id = null;

// Branding files are public (no auth required for login pages)
if ($type === 'branding') {
    $user_type = 'public';
    $user_id = 0;
} else {
    // Check admin authentication
    if (file_exists(__DIR__ . '/../includes/auth_check.php')) {
        require_once __DIR__ . '/../includes/auth_check.php';
        if (function_exists('is_logged_in') && is_logged_in()) {
            $user_type = 'admin';
            $user_id = current_user_id();
        }
    }

    // Check driver authentication if not admin
    if ($user_type === null && file_exists(__DIR__ . '/../includes/driver_auth.php')) {
        require_once __DIR__ . '/../includes/driver_auth.php';

        // Check if driver is logged in (session already started above)
        if (function_exists('driver_logged_in') && driver_logged_in()) {
            $user_type = 'driver';
            $user_id = current_driver_id();
        }
    }

    // Must be authenticated for non-branding files
    if ($user_type === null) {
        http_response_code(401);
        exit('Unauthorized');
    }
}

// Sanitize path - remove any directory traversal attempts
$path = str_replace(['..', "\0"], '', $path);
$path = ltrim($path, '/\\');

if ($path === '') {
    http_response_code(400);
    exit('Invalid path');
}

// Build absolute path
$storage_base = STORAGE_DIR;
$full_path = $storage_base . '/' . $type . '/' . $path;

// Resolve to real path and validate it's within storage directory
$real_path = realpath($full_path);
$real_base = realpath($storage_base . '/' . $type);

if ($real_path === false || $real_base === false) {
    http_response_code(404);
    exit('File not found');
}

// Security: ensure the resolved path is within the allowed directory
if (stripos($real_path, $real_base) !== 0) {
    http_response_code(403);
    exit('Access denied');
}

// Check file exists and is readable
if (!is_file($real_path) || !is_readable($real_path)) {
    http_response_code(404);
    exit('File not found');
}

// Access control for drivers - they can only access their own files
if ($user_type === 'driver' && $type === 'submissions') {
    require_once __DIR__ . '/../config/db.php';

    // Get driver's partner_id
    $stmt = $conn->prepare("SELECT partner_id FROM drivers WHERE id = ? LIMIT 1");
    $stmt->bind_param('i', $user_id);
    $stmt->execute();
    $driver = $stmt->get_result()->fetch_assoc();

    if (!$driver) {
        http_response_code(403);
        exit('Access denied');
    }

    // Check if the path contains the driver's partner_id
    $partner_id = $driver['partner_id'];
    if (stripos($path, $partner_id) === false) {
        http_response_code(403);
        exit('Access denied');
    }
}

if ($user_type === 'driver' && $type === 'docs') {
    require_once __DIR__ . '/../config/db.php';

    // Get driver's partner_id
    $stmt = $conn->prepare("SELECT partner_id FROM drivers WHERE id = ? LIMIT 1");
    $stmt->bind_param('i', $user_id);
    $stmt->execute();
    $driver = $stmt->get_result()->fetch_assoc();

    if (!$driver) {
        http_response_code(403);
        exit('Access denied');
    }

    // Check if the path contains the driver's partner_id (case-insensitive)
    // Path format: {partner_id}/{group_key}/{year}/{month}/{filename}
    $partner_id = $driver['partner_id'];
    $path_normalized = strtolower(trim($path, '/'));
    $partner_id_normalized = strtolower($partner_id);

    // Check if path starts with partner_id (allowing for URL encoding)
    if (stripos($path_normalized, $partner_id_normalized) !== 0) {
        http_response_code(403);
        exit('Access denied');
    }
}

// Detect MIME type
$finfo = finfo_open(FILEINFO_MIME_TYPE);
$mime = finfo_file($finfo, $real_path) ?: 'application/octet-stream';
finfo_close($finfo);

// Serve the file
header('Content-Type: ' . $mime);
header('Content-Length: ' . filesize($real_path));
header('Cache-Control: ' . ($type === 'branding' ? 'public, max-age=3600' : 'private, max-age=600'));
header('Content-Disposition: inline; filename="' . basename($real_path) . '"');

readfile($real_path);
exit;
