<?php
/**
 * migrate_uploads.php
 *
 * Migrates uploaded files from various legacy locations to /storage/ and updates database paths.
 *
 * Usage: php scripts/migrate_uploads.php [--dry-run]
 *
 * --dry-run: Show what would be done without making changes
 */

declare(strict_types=1);

require_once __DIR__ . '/../includes/constants.php';
require_once __DIR__ . '/../config/db.php';

$dry_run = in_array('--dry-run', $argv ?? []);

echo "=== Hisbuu Upload Migration Script ===\n";
echo $dry_run ? "MODE: DRY RUN (no changes will be made)\n" : "MODE: LIVE\n";
echo "Date: " . date('Y-m-d H:i:s') . "\n\n";

$project_root = realpath(__DIR__ . '/..');
$storage_dir = STORAGE_DIR;

echo "Project root: $project_root\n";
echo "Target storage: $storage_dir\n\n";

// Ensure storage directories exist
if (!$dry_run) {
    @mkdir($storage_dir . '/submissions', 0775, true);
    @mkdir($storage_dir . '/docs', 0775, true);
}

$stats = [
    'files_moved' => 0,
    'files_skipped' => 0,
    'files_failed' => 0,
    'db_updated' => 0,
    'db_failed' => 0,
];

/**
 * Find the actual file on disk given a database path
 */
function find_source_file(string $db_path, string $project_root): ?string {
    // Clean up the path
    $db_path = trim($db_path);

    // Extract just the filename for legacy paths
    $filename = basename($db_path);

    // Possible source locations to check
    $possible_sources = [];

    // 1. If path contains specific patterns, try those first
    if (preg_match('~/hisbuu/public/uploads/(.+)$~', $db_path, $m)) {
        $possible_sources[] = $project_root . '/public/uploads/' . $m[1];
    }
    if (preg_match('~^/?uploads/(.+)$~', $db_path, $m)) {
        $possible_sources[] = $project_root . '/public/uploads/' . $m[1];
    }

    // 2. Check public/storage/submissions (legacy location)
    $possible_sources[] = $project_root . '/public/storage/submissions/' . $filename;

    // 3. Check root storage/submissions
    $possible_sources[] = $project_root . '/storage/submissions/' . $filename;

    // 4. Check public/uploads with various subdirectories
    $possible_sources[] = $project_root . '/public/uploads/' . $filename;
    $possible_sources[] = $project_root . '/public/uploads/submissions/' . $filename;

    // 5. For paths with full structure, try direct
    if (preg_match('~driver_submissions/(.+)$~', $db_path, $m)) {
        $possible_sources[] = $project_root . '/public/uploads/driver_submissions/' . $m[1];
    }
    if (preg_match('~submissions/(.+)$~', $db_path, $m)) {
        $possible_sources[] = $project_root . '/public/uploads/submissions/' . $m[1];
    }

    foreach ($possible_sources as $source) {
        if (file_exists($source) && is_file($source)) {
            return $source;
        }
    }

    return null;
}

/**
 * Find source file for documents
 */
function find_doc_source(string $db_path, string $project_root): ?string {
    $db_path = trim($db_path);

    $possible_sources = [];

    // Extract relative path after uploads/docs/
    if (preg_match('~uploads/docs/(.+)$~', $db_path, $m)) {
        $possible_sources[] = $project_root . '/public/uploads/docs/' . $m[1];
    }

    // Direct path check
    $possible_sources[] = $project_root . '/public/' . ltrim($db_path, '/');
    $possible_sources[] = $project_root . '/public/uploads/docs/' . basename($db_path);

    foreach ($possible_sources as $source) {
        if (file_exists($source) && is_file($source)) {
            return $source;
        }
    }

    return null;
}

/**
 * Migrate a submission file
 */
function migrate_submission(string $db_path, bool $dry_run, array &$stats, string $project_root, string $storage_dir): ?string {
    // Skip if already migrated
    if (str_starts_with($db_path, 'storage/')) {
        return null;
    }

    $source = find_source_file($db_path, $project_root);

    if ($source === null) {
        echo "  SKIP: Source not found for: $db_path\n";
        $stats['files_skipped']++;
        return null;
    }

    $filename = basename($source);
    $target = $storage_dir . '/submissions/' . $filename;
    $new_db_path = 'storage/submissions/' . $filename;

    if (file_exists($target)) {
        echo "  EXISTS: $target (will update DB path)\n";
        $stats['files_skipped']++;
        return $new_db_path;
    }

    if ($dry_run) {
        echo "  WOULD MOVE: $source -> $target\n";
        $stats['files_moved']++;
        return $new_db_path;
    }

    // Move file
    if (rename($source, $target)) {
        echo "  MOVED: $source -> $target\n";
        $stats['files_moved']++;
        return $new_db_path;
    } else {
        // Try copy + delete if rename fails (cross-device)
        if (copy($source, $target)) {
            unlink($source);
            echo "  COPIED+DELETED: $source -> $target\n";
            $stats['files_moved']++;
            return $new_db_path;
        }
        echo "  FAIL: Could not move: $source\n";
        $stats['files_failed']++;
        return null;
    }
}

/**
 * Migrate a document file
 */
function migrate_document(string $db_path, bool $dry_run, array &$stats, string $project_root, string $storage_dir): ?string {
    // Skip if already migrated
    if (str_starts_with($db_path, 'storage/')) {
        return null;
    }

    $source = find_doc_source($db_path, $project_root);

    if ($source === null) {
        echo "  SKIP: Source not found for: $db_path\n";
        $stats['files_skipped']++;
        return null;
    }

    // Preserve directory structure for docs
    if (preg_match('~uploads/docs/(.+)$~', $db_path, $m)) {
        $rel_path = $m[1];
    } else {
        $rel_path = basename($source);
    }

    $target = $storage_dir . '/docs/' . $rel_path;
    $target_dir = dirname($target);
    $new_db_path = 'storage/docs/' . $rel_path;

    if (file_exists($target)) {
        echo "  EXISTS: $target (will update DB path)\n";
        $stats['files_skipped']++;
        return $new_db_path;
    }

    if ($dry_run) {
        echo "  WOULD MOVE: $source -> $target\n";
        $stats['files_moved']++;
        return $new_db_path;
    }

    // Create target directory
    if (!is_dir($target_dir)) {
        @mkdir($target_dir, 0775, true);
    }

    // Move file
    if (rename($source, $target)) {
        echo "  MOVED: $source -> $target\n";
        $stats['files_moved']++;
        return $new_db_path;
    } else {
        if (copy($source, $target)) {
            unlink($source);
            echo "  COPIED+DELETED: $source -> $target\n";
            $stats['files_moved']++;
            return $new_db_path;
        }
        echo "  FAIL: Could not move: $source\n";
        $stats['files_failed']++;
        return null;
    }
}

// ============================================
// 1. Migrate submissions table
// ============================================
echo "\n--- Migrating submissions ---\n";

$result = $conn->query("SELECT id, ss_orders, ss_deposit_1, ss_deposit_2 FROM submissions");
while ($row = $result->fetch_assoc()) {
    $id = (int)$row['id'];
    $updates = [];

    foreach (['ss_orders', 'ss_deposit_1', 'ss_deposit_2'] as $col) {
        $path = $row[$col];
        if (empty($path)) continue;

        // Skip if already migrated
        if (str_starts_with($path, 'storage/')) {
            continue;
        }

        echo "Submission #$id, $col: $path\n";
        $new_path = migrate_submission($path, $dry_run, $stats, $project_root, $storage_dir);

        if ($new_path !== null) {
            $updates[$col] = $new_path;
        }
    }

    if (!empty($updates)) {
        if (!$dry_run) {
            $set_parts = [];
            $values = [];
            $types = '';
            foreach ($updates as $col => $val) {
                $set_parts[] = "$col = ?";
                $values[] = $val;
                $types .= 's';
            }
            $values[] = $id;
            $types .= 'i';

            $sql = "UPDATE submissions SET " . implode(', ', $set_parts) . " WHERE id = ?";
            $stmt = $conn->prepare($sql);
            $stmt->bind_param($types, ...$values);
            if ($stmt->execute()) {
                echo "  DB UPDATED: Submission #$id\n";
                $stats['db_updated']++;
            } else {
                echo "  DB FAIL: Submission #$id\n";
                $stats['db_failed']++;
            }
        } else {
            echo "  WOULD UPDATE DB: Submission #$id -> " . implode(', ', array_values($updates)) . "\n";
            $stats['db_updated']++;
        }
    }
}

// ============================================
// 2. Migrate driver_documents table
// ============================================
echo "\n--- Migrating driver_documents ---\n";

$result = $conn->query("SELECT id, file_path FROM driver_documents WHERE file_path IS NOT NULL AND file_path != ''");
while ($row = $result->fetch_assoc()) {
    $id = (int)$row['id'];
    $path = $row['file_path'];

    // Skip if already migrated
    if (str_starts_with($path, 'storage/')) {
        continue;
    }

    echo "Document #$id: $path\n";
    $new_path = migrate_document($path, $dry_run, $stats, $project_root, $storage_dir);

    if ($new_path !== null) {
        if (!$dry_run) {
            $stmt = $conn->prepare("UPDATE driver_documents SET file_path = ? WHERE id = ?");
            $stmt->bind_param('si', $new_path, $id);
            if ($stmt->execute()) {
                echo "  DB UPDATED: Document #$id\n";
                $stats['db_updated']++;
            } else {
                echo "  DB FAIL: Document #$id\n";
                $stats['db_failed']++;
            }
        } else {
            echo "  WOULD UPDATE DB: Document #$id -> $new_path\n";
            $stats['db_updated']++;
        }
    }
}

// ============================================
// Summary
// ============================================
echo "\n=== Migration Summary ===\n";
echo "Files moved:    {$stats['files_moved']}\n";
echo "Files skipped:  {$stats['files_skipped']}\n";
echo "Files failed:   {$stats['files_failed']}\n";
echo "DB updated:     {$stats['db_updated']}\n";
echo "DB failed:      {$stats['db_failed']}\n";

if ($dry_run) {
    echo "\nThis was a DRY RUN. Run without --dry-run to apply changes.\n";
} else {
    echo "\nMigration complete. You can now delete:\n";
    echo "  - public/storage/\n";
    echo "  - public/uploads/\n";
}

echo "\nDone.\n";
